#!/bin/bash

# Script de preparation d'une machine de contrôle Ansible pour GCV6
# Utilisation: ./setup-control-machine.sh
# Prerequis: Executer en tant que root

set -euo pipefail

SCRIPT_NAME="$(basename "$0")"
LOG_DIR="/var/log"
LOG_FILE="$LOG_DIR/ansible-control-setup.log"
PYTHON_VERSION="3.11.10"
LOCAL_DIR="/home/local"
PYTHON_DIR="$LOCAL_DIR/python-3.11.10"
VENV_DIR="$LOCAL_DIR/venv-ansible"
ADMINISTRATOR_USER="administrator"

# Configuration couleurs pour l'affichage
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Fonction de logging
log_message() {
    local level="$1"
    local message="$2"
    local timestamp=$(date '+%Y-%m-%d %H:%M:%S')
    echo "[$timestamp] [$level] $message" >> "$LOG_FILE"
    
    case "$level" in
        "INFO")
            echo -e "${GREEN}[INFO]${NC} $message"
            ;;
        "WARN")
            echo -e "${YELLOW}[WARN]${NC} $message"
            ;;
        "ERROR")
            echo -e "${RED}[ERROR]${NC} $message"
            ;;
        "DEBUG")
            echo -e "${BLUE}[DEBUG]${NC} $message"
            ;;
    esac
}

# Fonction de verification des prerequis
check_prerequisites() {
    log_message "INFO" "Verification des prerequis..."
    
    # Verification root
    if [ "$EUID" -ne 0 ]; then
        log_message "ERROR" "Ce script doit etre execute en tant que root"
        exit 1
    fi
    
    # Verification version CentOS 6
    if [ -f /etc/centos-release ]; then
        if ! grep -q "release 6" /etc/centos-release; then
            log_message "ERROR" "Ce script necessite CentOS 6"
            exit 1
        fi
    else
        log_message "ERROR" "Fichier /etc/centos-release introuvable, impossible de verifier la version"
        exit 1
    fi
    
    ## Verification connectivite Internet
    #if ! ping -c 1 8.8.8.8 >/dev/null 2>&1; then
    #    log_message "WARN" "Pas de connectivite Internet detectee - certaines etapes pourraient echouer"
    #fi
    
    log_message "INFO" "Prerequis verifies avec succes"
}

# Creation de l'utilisateur administrator
create_administrator_user() {
    log_message "INFO" "Creation de l'utilisateur $ADMINISTRATOR_USER..."

    # Verifier si l'utilisateur existe deja
    if id "$ADMINISTRATOR_USER" &>/dev/null; then
        log_message "WARN" "L'utilisateur $ADMINISTRATOR_USER existe deja"
        return 0
    fi

    # Creer l'utilisateur avec groupe wheel
    useradd -m -G wheel -s /bin/bash "$ADMINISTRATOR_USER"
    usermod -aG gestcom "$ADMINISTRATOR_USER"
    
    ## Definir un mot de passe temporaire
    #echo "admin123!" | passwd --stdin "$ADMINISTRATOR_USER"
    ## Forcer le changement de mot de passe a la premiere connexion
    #chage -d 0 "$ADMINISTRATOR_USER"
    
    log_message "INFO" "Utilisateur $ADMINISTRATOR_USER cree avec succes"
    #log_message "WARN" "Mot de passe temporaire: admin123! (changement obligatoire a la premiere connexion)"
}

# Configuration du dossier utilisateur securise
configure_user_directory() {
    log_message "INFO" "Configuration du dossier utilisateur securise..."
    
    local user_home="/home/$ADMINISTRATOR_USER"
    
    # Permissions restrictives sur le dossier home
    chmod 700 "$user_home"
    chown "$ADMINISTRATOR_USER:$ADMINISTRATOR_USER" "$user_home"
    
    # Creation du dossier .ssh
    local ssh_dir="$user_home/.ssh"
    mkdir -p "$ssh_dir"
    chmod 700 "$ssh_dir"
    chown "$ADMINISTRATOR_USER:$ADMINISTRATOR_USER" "$ssh_dir"
    
    # Creation du dossier .scripts
    local scripts_dir="$user_home/.scripts"
    mkdir -p "$scripts_dir"
    chmod 755 "$scripts_dir"
    chown "$ADMINISTRATOR_USER:$ADMINISTRATOR_USER" "$scripts_dir"
    
    log_message "INFO" "Dossier utilisateur configure avec permissions restrictives"
}

# Generation de la cle SSH
generate_ssh_key() {
    log_message "INFO" "Generation de la cle SSH avec mot de passe..."
    
    local user_home="/home/$ADMINISTRATOR_USER"
    local ssh_dir="$user_home/.ssh"
    local key_file="$ssh_dir/id_rsa"
    
    # Generation cle SSH
    su -c "ssh-keygen -t rsa -b 4096 -f $key_file -N ansible2024! -C $ADMINISTRATOR_USER@ansible-control" "$ADMINISTRATOR_USER"
    su -c "cp $key_file.pub $ssh_dir/authorized_keys" "$ADMINISTRATOR_USER"
    su -c "chmod 600 $ssh_dir/authorized_keys" "$ADMINISTRATOR_USER"

    log_message "INFO" "Cle SSH generee avec succes"
    log_message "WARN" "Passphrase de la cle SSH: ansible2024!"
}

# Installation de Python 3.11 et environnement Ansible
install_python311() {
    log_message "INFO" "Decompression de Python et environnement Ansible"

    # Creer le repertoire local
    mkdir -p "$LOCAL_DIR"
    
    # Decompression de Python 3.11
    tar xJf ./files/python3.11.10-centos6-20250927.tar.xz -C "$LOCAL_DIR"
    
    # Decompression de l'environnement virtuel Ansible
    tar xJf ./files/python3.11.10-centos6-ansible-20250927.tar.xz -C "$LOCAL_DIR"
    chown -R "$ADMINISTRATOR_USER:$ADMINISTRATOR_USER" "$LOCAL_DIR/venv.ansible"
    
# Script d'activation
cat > "/home/$ADMINISTRATOR_USER/.scripts/activate-ansible.sh" << 'EOF'
#!/bin/bash
export PATH=/home/local/python-3.11.10/bin:$PATH
source /home/local/venv.ansible/bin/activate
echo "Environnement Ansible active"
echo "Version Python: \$(python --version)"
echo "Version Ansible: \$(ansible --version | head -n1)"
EOF

    chmod +x "/home/$ADMINISTRATOR_USER/.scripts/activate-ansible.sh"
    chown "$ADMINISTRATOR_USER:$ADMINISTRATOR_USER" "/home/$ADMINISTRATOR_USER/.scripts/activate-ansible.sh"

    log_message "INFO" "Python $PYTHON_VERSION et environnement Ansible installes dans $LOCAL_DIR"
}


# Configuration finale
final_configuration() {
    log_message "INFO" "Configuration finale..."
    
    # Ajouter l'utilisateur aux sudoers avec NOPASSWD pour certaines commandes
 cat > "/etc/sudoers.d/$ADMINISTRATOR_USER" << EOF
# Permissions pour l'utilisateur administrator
$ADMINISTRATOR_USER ALL=(ALL) NOPASSWD: /home/local/venv.ansible/bin/ansible-playbook
$ADMINISTRATOR_USER ALL=(ALL) ALL
EOF
    
    chmod 440 "/etc/sudoers.d/$ADMINISTRATOR_USER"
    
    # Configuration du profil utilisateur avec activation automatique
    # Ajout alias dans bashrc
    echo "#alias activate-ansible='source ~/.scripts/activate-ansible.sh'" >> "/home/${ADMINISTRATOR_USER}/.bashrc"
    echo "source ~/.scripts/activate-ansible.sh" >> "/home/${ADMINISTRATOR_USER}/.bashrc"

    chown "$ADMINISTRATOR_USER:$ADMINISTRATOR_USER" "/home/$ADMINISTRATOR_USER/.bashrc"

    log_message "INFO" "Configuration finale terminee"
}

# Affichage du resume
display_summary() {
    log_message "INFO" "=== ReSUMe DE L'INSTALLATION ==="
    log_message "INFO" "Machine de contrôle Ansible configuree avec succes!"
    log_message "INFO" ""
    log_message "INFO" "Utilisateur cree: $ADMINISTRATOR_USER"
    log_message "INFO" "Mot de passe temporaire: admin123! (changement requis)"
    log_message "INFO" "Cle SSH: /home/$ADMINISTRATOR_USER/.ssh/id_rsa"
    log_message "INFO" "Passphrase SSH: ansible2024!"
    log_message "INFO" ""
    log_message "INFO" "Python 3.11: $PYTHON_DIR"
    log_message "INFO" "Environnement Ansible: $LOCAL_DIR/venv.ansible"
    log_message "INFO" "Script d'activation: $LOCAL_DIR/activate-ansible.sh"
    log_message "INFO" ""
    log_message "INFO" "Pour commencer:"
    log_message "INFO" "1. Connectez-vous avec l'utilisateur $ADMINISTRATOR_USER"
    log_message "INFO" "2. Changez le mot de passe a la premiere connexion"
    log_message "INFO" "3. Executez: source /home/local/activate-ansible.sh"
    log_message "INFO" "4. Ou utilisez l'alias: activate-ansible"
    log_message "INFO" ""
    log_message "INFO" "Logs complets disponibles dans: $LOG_FILE"
}

# Fonction principale
main() {
    log_message "INFO" "=== DeMARRAGE DU SCRIPT DE CONFIGURATION ANSIBLE ==="
    
    check_prerequisites
    create_administrator_user
    configure_user_directory
    generate_ssh_key
    install_python311
    final_configuration
    display_summary
    
    log_message "INFO" "Script termine avec succes!"
}

# Gestion des signaux
trap 'log_message "ERROR" "Script interrompu par l'\''utilisateur"; exit 130' INT TERM

# Execution principale
main "$@"