#!/usr/bin/env python3
import sqlite3
import typer
from datetime import datetime
from typing import Optional

def format_date_time(date_str: str, time_str: str) -> str:
    """Formate la date et l'heure pour l'affichage"""
    try:
        # Convertir YYYYMMDD HHMMSS en format lisible
        date_obj = datetime.strptime(f"{date_str} {time_str}", "%Y%m%d %H%M%S")
        return date_obj.strftime("%d/%m/%Y %H:%M:%S")
    except ValueError:
        return f"{date_str} {time_str}"

def list_deployments(
    db_path: str = typer.Argument(..., help="Chemin vers la base de données SQLite"),
    limit: int = typer.Option(10, "--limit", "-l", help="Nombre d'installations à afficher")
):
    """Afficher les dernières installations sous forme de tableau"""
    try:
        conn = sqlite3.connect(db_path)
        cursor = conn.cursor()
        
        # Récupérer les dernières installations, triées par date/heure décroissante
        cursor.execute("""
            SELECT version, date, time, created_at 
            FROM deployments 
            ORDER BY date DESC, time DESC 
            LIMIT ?
        """, (limit,))
        
        deployments = cursor.fetchall()
        conn.close()
        
        if not deployments:
            typer.echo("Aucune installation trouvée dans la base de données.")
            return
        
        # Affichage du tableau
        typer.echo("\n" + "="*70)
        typer.echo("HISTORIQUE DES INSTALLATIONS GCV")
        typer.echo("="*70)
        typer.echo(f"{'#':<3} {'Version':<20} {'Date/Heure':<20} {'Enregistré le':<20}")
        typer.echo("-"*70)
        
        for i, (version, date, time, created_at) in enumerate(deployments, 1):
            formatted_datetime = format_date_time(date, time)
            # Formatter created_at si présent
            if created_at:
                try:
                    created_obj = datetime.fromisoformat(created_at.replace('Z', '+00:00'))
                    created_formatted = created_obj.strftime("%d/%m/%Y %H:%M")
                except:
                    created_formatted = created_at[:16] if created_at else "N/A"
            else:
                created_formatted = "N/A"
            
            typer.echo(f"{i:<3} {version:<20} {formatted_datetime:<20} {created_formatted:<20}")
        
        typer.echo("-"*70)
        typer.echo(f"Total: {len(deployments)} installation(s) affichée(s)")
        typer.echo("="*70 + "\n")
        
    except sqlite3.Error as e:
        typer.echo(f"Erreur SQLite: {e}", err=True)
        raise typer.Exit(1)
    except FileNotFoundError:
        typer.echo(f"Base de données non trouvée: {db_path}", err=True)
        raise typer.Exit(1)
    except Exception as e:
        typer.echo(f"Erreur: {e}", err=True)
        raise typer.Exit(1)

if __name__ == '__main__':
    typer.run(list_deployments)