#!/bin/bash
#
# find_sparse_files.sh
# Détecte les fichiers sparse dans une base Informix SE
# et identifie s'il s'agit de fichiers data (.dat) ou index (.idx)
#
# Usage: ./find_sparse_files.sh <répertoire_dbs> [seuil_ratio]
#        seuil_ratio: ratio min taille_apparente/taille_reelle pour considérer sparse (défaut: 1.5)

REPERTOIRE="${1:-.}"
SEUIL="${2:-1.5}"

if [[ ! -d "$REPERTOIRE" ]]; then
    echo "Erreur: $REPERTOIRE n'est pas un répertoire valide"
    exit 1
fi

echo "=========================================="
echo "Recherche de fichiers sparse"
echo "Répertoire: $REPERTOIRE"
echo "Seuil ratio: $SEUIL"
echo "=========================================="
echo ""

printf "%-50s %12s %12s %8s %s\n" "FICHIER" "APPARENT" "REEL" "RATIO" "TYPE"
printf "%-50s %12s %12s %8s %s\n" "-------" "--------" "----" "-----" "----"

find "$REPERTOIRE" -type f \( -name "*.dat" -o -name "*.idx" \) | while read -r fichier; do
    # Taille apparente (ls) en octets
    taille_apparente=$(stat -c %s "$fichier" 2>/dev/null)
    
    # Taille réelle sur disque (blocs * 512) en octets
    blocs=$(stat -c %b "$fichier" 2>/dev/null)
    taille_reelle=$((blocs * 512))
    
    # Éviter division par zéro
    if [[ $taille_reelle -eq 0 ]]; then
        if [[ $taille_apparente -gt 0 ]]; then
            ratio="INF"
        else
            continue
        fi
    else
        ratio=$(awk "BEGIN {printf \"%.1f\", $taille_apparente / $taille_reelle}")
    fi
    
    # Vérifier si sparse (ratio > seuil)
    is_sparse=0
    if [[ "$ratio" == "INF" ]]; then
        is_sparse=1
    else
        is_sparse=$(awk "BEGIN {print ($ratio > $SEUIL) ? 1 : 0}")
    fi
    
    if [[ $is_sparse -eq 1 ]]; then
        # Déterminer le type de fichier
        extension="${fichier##*.}"
        case "$extension" in
            dat) type="DATA" ;;
            idx) type="INDEX" ;;
            *)   type="AUTRE" ;;
        esac
        
        # Formater les tailles en Go
        apparent_go=$(awk "BEGIN {printf \"%.2f Go\", $taille_apparente / 1024 / 1024 / 1024}")
        reel_go=$(awk "BEGIN {printf \"%.2f Go\", $taille_reelle / 1024 / 1024 / 1024}")
        
        printf "%-50s %12s %12s %8s %s\n" "$fichier" "$apparent_go" "$reel_go" "${ratio}x" "$type"
    fi
done

echo ""
echo "=========================================="
echo "Terminé"
echo "=========================================="

