#!/bin/bash

# Script pour générer automatiquement un script de réparation de table
# Usage: ./create_repair_bash.sh <table_name>

set -e

# Couleurs
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

if [ $# -ne 1 ]; then
    echo -e "${RED}Usage: $0 <table_name>${NC}"
    echo -e "${YELLOW}Exemple: $0 gc_art2${NC}"
    exit 1
fi

TABLE_NAME="$1"
DBNAME="gc"
# Enlever le préfixe gc_ si présent pour le nom du fichier
OUTPUT_NAME="${TABLE_NAME#gc_}"
OUTPUT_FILE="repair_${OUTPUT_NAME}.sh"

# Vérifier les variables d'environnement
if [ -z "$INFORMIXDIR" ] || [ -z "$INFORMIXSERVER" ]; then
    echo -e "${RED}Les variables d'environnement INFORMIXDIR et INFORMIXSERVER doivent être définies.${NC}"
    exit 1
fi

echo -e "${CYAN}>> Récupération des informations de la table $TABLE_NAME...${NC}"

# Utiliser dbschema pour obtenir les définitions d'index
SCHEMA_FILE=$(mktemp)
dbschema -d "$DBNAME" -t "$TABLE_NAME" -ss > "$SCHEMA_FILE" 2>/dev/null

# Extraire les commandes CREATE INDEX (en gérant les lignes multiples)
# On remplace les retours à la ligne dans les commandes CREATE INDEX par des espaces
CREATE_COMMANDS=$(awk '
/create.*index/,/;/ {
    # Enlever les retours à la ligne et concaténer
    if (/;/) {
        line = line " " $0
        # Nettoyer les espaces multiples et afficher
        gsub(/[[:space:]]+/, " ", line)
        gsub(/^ /, "", line)
        gsub(/ ;/, ";", line)
        print line
        line = ""
    } else {
        gsub(/^[[:space:]]+/, "", $0)  # Enlever espaces de début
        line = line " " $0
    }
}' "$SCHEMA_FILE" | sed 's/"[^"]*"\.//g')

# Extraire les noms d'index
INDEX_NAMES=$(echo "$CREATE_COMMANDS" | grep -oP 'index\s+\K[a-zA-Z_][a-zA-Z0-9_]*' | sort -u)

if [ -z "$INDEX_NAMES" ]; then
    echo -e "${RED}Aucun index trouvé pour la table $TABLE_NAME${NC}"
    rm -f "$SCHEMA_FILE"
    exit 1
fi

echo -e "${GREEN}Index trouvés:${NC}"
echo "$INDEX_NAMES"
echo ""
echo -e "${GREEN}Commandes CREATE INDEX:${NC}"
echo "$CREATE_COMMANDS"
echo ""


# Générer le script de réparation
cat > "$OUTPUT_FILE" <<'SCRIPT_HEADER'
#!/bin/bash

# Couleurs
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

# Configuration
DBNAME="gc"
TABLE="__TABLE_NAME__"
SCRIPT_HEADER

# Ajouter les noms d'index comme variables
INDEX_COUNT=1
for idx in $INDEX_NAMES; do
    echo "INDEXNAME${INDEX_COUNT}=\"$idx\"" >> "$OUTPUT_FILE"
    INDEX_COUNT=$((INDEX_COUNT + 1))
done

cat >> "$OUTPUT_FILE" <<'SCRIPT_BODY'
ISQL="isql"
BATCH_SIZE=40  # Nombre de ROWID à traiter par lot

SCRIPT_NAME=$(basename $0 .sh)

if [ -z "$INFORMIXDIR" ] || [ -z "$INFORMIXSERVER" ] || [ -z "$DBPATH" ]; then
    echo -e "${RED}Erreur: Les variables d'environnement INFORMIXDIR, INFORMIXSERVER et DBPATH doivent etre definies.${NC}"
    exit 1
fi

if [ ! -d "$DBPATH" ]; then
    echo -e "${RED}Erreur: Le chemin de la base de donnees DBPATH=$DBPATH n'existe pas.${NC}"
    exit 1
fi

DBPATHNAME=$(basename $DBPATH)
# Fichiers temporaires
ERRORLOG=/home/log/${DBPATHNAME}/${SCRIPT_NAME}.log
SQLFILE=/home/log/${DBPATHNAME}/${SCRIPT_NAME}.sql
ROWID_FILE=/home/log/${DBPATHNAME}/${SCRIPT_NAME}.rowids
SCHEMA_BEFORE=/home/log/${DBPATHNAME}/${SCRIPT_NAME}_schema_before.sql
SCHEMA_AFTER=/home/log/${DBPATHNAME}/${SCRIPT_NAME}_schema_after.sql

echo -e "${CYAN}>> Etape 1 : Sauvegarde du schema avant modification...${NC}"

# Sauvegarder le schéma de la table avant modification
dbschema -d $DBNAME -t $TABLE -ss > "$SCHEMA_BEFORE" 2>/dev/null
echo -e "${GREEN}  ✓ Schema sauvegarde : $SCHEMA_BEFORE${NC}"

echo -e "${CYAN}>> Etape 2 : Repair de la table $TABLE et capture des erreurs...${NC}"

# Execution du REPAIR TABLE et redirection de stderr vers ERRORLOG
echo "repair table $TABLE" | $ISQL gc > "$ERRORLOG" 2>&1

echo -e "${CYAN}>> Etape 3 : Extraction des ROWID des doublons...${NC}"

# Extraire les rowid à partir de lignes comme : ERROR: duplicate key value, record 722893
grep -E "ERROR: duplicate key value, record [0-9]+" "$ERRORLOG" \
    | grep -oE "[0-9]+" > "$ROWID_FILE"

if [ ! -s "$ROWID_FILE" ]; then
    echo -e "${GREEN}  ✓ Aucun doublon trouve.${NC}"
    rm -f "$ERRORLOG" "$SQLFILE" "$ROWID_FILE"
    exit 0
fi

ROWID_COUNT=$(wc -l < "$ROWID_FILE")
echo -e "${YELLOW}  ⚠ Nombre de doublons detectes : $ROWID_COUNT${NC}"

# Generer le fichier SQL pour supprimer les index
cat <<EOF > "$SQLFILE"
-- Supprimer les index existants
SCRIPT_BODY

# Ajouter les commandes DROP INDEX
INDEX_COUNT=1
for idx in $INDEX_NAMES; do
    echo "DROP INDEX \$INDEXNAME${INDEX_COUNT};" >> "$OUTPUT_FILE"
    INDEX_COUNT=$((INDEX_COUNT + 1))
done

cat >> "$OUTPUT_FILE" <<'SCRIPT_MIDDLE1'
EOF

echo -e "${CYAN}>> Etape 4 : Suppression des index...${NC}"
$ISQL $DBNAME < "$SQLFILE"

# Supprimer les doublons par lots
echo -e "${CYAN}>> Etape 5 : Suppression des doublons (par lots de $BATCH_SIZE)...${NC}"

BATCH_NUM=0
TOTAL_DELETED=0
ROWID_BATCH=()

while IFS= read -r line; do
    ROWID_BATCH+=("$line")

    # Si le batch est plein
    if [ ${#ROWID_BATCH[@]} -eq $BATCH_SIZE ]; then
        BATCH_NUM=$((BATCH_NUM + 1))
        ROWID_LIST=$(IFS=,; echo "${ROWID_BATCH[*]}")

        echo -e "${BLUE}  Batch $BATCH_NUM : suppression de ${#ROWID_BATCH[@]} lignes...${NC}"

        cat <<EOFDELETE > "$SQLFILE"
DELETE FROM $TABLE WHERE rowid IN ($ROWID_LIST);
EOFDELETE
        $ISQL $DBNAME < "$SQLFILE"

        TOTAL_DELETED=$((TOTAL_DELETED + ${#ROWID_BATCH[@]}))
        ROWID_BATCH=()
    fi
done < "$ROWID_FILE"

# Traiter le dernier batch s'il reste des éléments
if [ ${#ROWID_BATCH[@]} -gt 0 ]; then
    BATCH_NUM=$((BATCH_NUM + 1))
    ROWID_LIST=$(IFS=,; echo "${ROWID_BATCH[*]}")

    echo -e "${BLUE}  Batch $BATCH_NUM (final) : suppression de ${#ROWID_BATCH[@]} lignes...${NC}"

    cat <<EOFDELETE > "$SQLFILE"
DELETE FROM $TABLE WHERE rowid IN ($ROWID_LIST);
EOFDELETE
    $ISQL $DBNAME < "$SQLFILE"

    TOTAL_DELETED=$((TOTAL_DELETED + ${#ROWID_BATCH[@]}))
fi

echo -e "${GREEN}  ✓ Total supprime : $TOTAL_DELETED lignes${NC}"

# Recreer les index
echo -e "${CYAN}>> Etape 6 : Recreation des index...${NC}"
cat <<EOF > "$SQLFILE"
-- Recreer les index
SCRIPT_MIDDLE1

# Ajouter les commandes CREATE INDEX
echo "$CREATE_COMMANDS" | sed 's/;$/;/' >> "$OUTPUT_FILE"

cat >> "$OUTPUT_FILE" <<'SCRIPT_FOOTER'
EOF

$ISQL $DBNAME < "$SQLFILE"

echo -e "${CYAN}>> Etape 7 : Sauvegarde du schema après modification...${NC}"

# Sauvegarder le schéma de la table après modification
dbschema -d $DBNAME -t $TABLE -ss > "$SCHEMA_AFTER" 2>/dev/null
echo -e "${GREEN}  ✓ Schema sauvegarde : $SCHEMA_AFTER${NC}"

# Comparer les schémas
echo -e "${CYAN}>> Etape 8 : Verification des index...${NC}"

# Extraire uniquement les lignes CREATE INDEX des deux fichiers
INDEXES_BEFORE=$(grep -i "create.*index" "$SCHEMA_BEFORE" | sort)
INDEXES_AFTER=$(grep -i "create.*index" "$SCHEMA_AFTER" | sort)

if [ "$INDEXES_BEFORE" == "$INDEXES_AFTER" ]; then
    echo -e "${GREEN}  ✓ Les index ont ete correctement recrees${NC}"
else
    echo -e "${RED}  ⚠ ATTENTION : Les index ne correspondent pas exactement${NC}"
    echo -e "${YELLOW}  Utilisez 'diff $SCHEMA_BEFORE $SCHEMA_AFTER' pour voir les differences${NC}"
fi

# Nettoyage
echo -e "${CYAN}>> Etape 9 : Recapitulatif...${NC}"
echo -e "  Erreurs repair    : ${YELLOW}${ERRORLOG}${NC}"
echo -e "  Requetes SQL      : ${YELLOW}${SQLFILE}${NC}"
echo -e "  ROWIDs doublons   : ${YELLOW}${ROWID_FILE}${NC}"
echo -e "  Schema avant      : ${YELLOW}${SCHEMA_BEFORE}${NC}"
echo -e "  Schema apres      : ${YELLOW}${SCHEMA_AFTER}${NC}"
echo ""
echo -e "${GREEN}✓ Script termine avec succes.${NC}"
SCRIPT_FOOTER

# Remplacer __TABLE_NAME__ par le vrai nom
sed -i "s/__TABLE_NAME__/$TABLE_NAME/g" "$OUTPUT_FILE"

# Rendre le script exécutable
chmod +x "$OUTPUT_FILE"

# Nettoyage
rm -f "$SCHEMA_FILE"

echo ""
echo -e "${GREEN}✓ Script de réparation généré : ${YELLOW}$OUTPUT_FILE${NC}"
echo ""
echo -e "${CYAN}Caractéristiques:${NC}"
echo -e "  ${BLUE}•${NC} Table: ${YELLOW}$TABLE_NAME${NC}"
echo -e "  ${BLUE}•${NC} Nombre d'index: ${YELLOW}$(echo "$INDEX_NAMES" | wc -w)${NC}"
echo -e "  ${BLUE}•${NC} Traitement par lots de ${YELLOW}40 ROWIDs${NC}"
echo ""
echo -e "${CYAN}Pour l'utiliser:${NC}"
echo -e "  ${YELLOW}./$OUTPUT_FILE${NC}"
