/**
 * @file download_files.c
 * @brief Programme de telechargement de fichiers via SFTP utilisant libcurl
 *
 * Conversion du script Python download_files.py en C avec support SFTP,
 * interface CLI avec saisie securisee de mot de passe et parsing d'arguments avec GLib.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <unistd.h>
#include <termios.h>
#include <sys/stat.h>
#include <curl/curl.h>
#include <regex.h>
#include <errno.h>
#include <glib.h>
#include <gio/gio.h>
#include <openssl/md5.h>

#define MAX_PATH 512
#define MAX_HOST 256
#define MAX_USER 128
#define MAX_PASS 128
#define MAX_VERSION 64
#define MAX_FILES 100
#define MAX_FILENAME 256

typedef struct
{
    char host[MAX_HOST];
    char user[MAX_USER];
    char pass[MAX_PASS];
    gboolean pass_provided;
} Config;

typedef struct
{
    char **filenames;
    int count;
    int capacity;
} FileList;

/* Options globales */
static gchar *opt_osname = NULL;
static gchar *opt_dir = NULL;
static gchar *opt_config = NULL;
static gboolean opt_list = FALSE;
static gboolean opt_version_only = FALSE;
static gboolean opt_verbose = FALSE;

/* ============================================================================
 * LISTE DES FICHIERS A TELECHARGER
 * ============================================================================
 * Modifier ces listes pour ajouter/supprimer des fichiers à télécharger
 * avec la commande 'download'
 */

/* Fichiers relatifs au répertoire de l'OS (ex: centos6/, centos9/)
 * Ces fichiers seront téléchargés depuis: sftp://host/{osname}/{filename}
 *
 * Template disponible: {osname} sera remplacé par la valeur de --osname
 * Exemple: "curl-{osname}.tar.bz2" devient "curl-centos6.tar.bz2" pour centos6
 */
static const char *CENTOS6ONLY_FILES_TO_DOWNLOAD[] = {
    "curl-7.76_centos6.v2.tar.bz2",
    "openssl-3.2.1_centos6.v2.tar.bz2",
    "sqlite3_centos6.v2.tar.bz2",
    "emc2-rpms.tar.xz"};
static const char *FILES_TO_DOWNLOAD[] = {
    "curl-7.76_{osname}.v2.tar.bz2",
    "openssl-3.2.1_{osname}.v2.tar.bz2",
    "sqlite3_{osname}.v2.tar.bz2",
    "python3.11.10-{osname}-20250927.tar.xz",
    "venv-python-{osname}.tar.xz",
    "python3.11.10-{osname}-ansible-20250927.tar.xz",
    "emc2-rpms.tar.xz"};
static const int NUM_FILES_TO_DOWNLOAD = sizeof(FILES_TO_DOWNLOAD) / sizeof(FILES_TO_DOWNLOAD[0]);

/* Fichiers avec chemins absolus (indépendants de l'OS)
 * Format: "chemin/distant/complet/fichier.tar.xz"
 * Ces fichiers seront téléchargés depuis: sftp://host/{chemin_complet}
 * Exemple: "/shared/tools/common-tools.tar.xz"
 */
static const char *FILES_ABSOLUTE_PATHS[] = {
    // Ajoutez vos fichiers avec chemins absolus ici
    // Exemple: "/shared/dependencies/libfoo.tar.xz",
    "/centos6/download_files-centos6",
    "/centos9/download_files-centos9",
};
static const int NUM_FILES_ABSOLUTE = sizeof(FILES_ABSOLUTE_PATHS) / sizeof(FILES_ABSOLUTE_PATHS[0]);

/* Prototypes */
static bool load_config(Config *config);
static bool save_default_config(const char *filepath);
static char *read_password(const char *prompt);
static char *detect_osname(void);
static char *replace_template(const char *template_str, const char *osname);
static bool download_file(const Config *config, const char *remote_path,
                          const char *local_path, const char *filename);
static bool upload_file(const Config *config, const char *remote_path,
                        const char *local_file);
static size_t write_callback(void *ptr, size_t size, size_t nmemb, FILE *stream);
static size_t read_callback(void *ptr, size_t size, size_t nmemb, FILE *stream);
static bool ensure_directory(const char *path);
static FileList *create_file_list(void);
static void add_file_to_list(FileList *list, const char *filename);
static void free_file_list(FileList *list);
static bool list_remote_files(const Config *config, const char *remote_path, FileList *list);
static char *find_latest_gc_archive(FileList *list, const char *osname, char *version_out);
static int cmd_download(const char *osname, const char *local_dir, bool list_only);
static int cmd_latest(const char *osname, const char *local_dir, bool version_only);
static int cmd_version(const char *version_arg, const char *osname, const char *local_dir);
static int cmd_upload(const char *osname, int num_files, char **files);
static int cmd_down(const char *osname, const char *local_dir, int num_files, char **files);
static int cmd_check(const char *osname);
static int progress_callback(void *clientp, curl_off_t dltotal, curl_off_t dlnow, curl_off_t ultotal, curl_off_t ulnow);
static bool compute_md5(const char *filename, char *output_hash);
static bool verify_md5(const char *filepath, const char *expected_md5);
static bool download_and_verify_file(const Config *config, const char *remote_path,
                                     const char *local_path, const char *filename);

/**
 * @brief Charge la configuration depuis le fichier spécifié ou ./config.ini par défaut
 */
static bool load_config(Config *config)
{
    char filepath[MAX_PATH];

    // Utiliser le chemin fourni via --config ou le chemin par défaut
    if (opt_config != NULL && strlen(opt_config) > 0)
    {
        snprintf(filepath, sizeof(filepath), "%s", opt_config);
    }
    else
    {
        snprintf(filepath, sizeof(filepath), "./config.ini");
    }

    config->pass_provided = false;

    FILE *fp = fopen(filepath, "r");
    if (!fp)
    {
        g_printerr("Fichier de configuration non trouve: %s\n", filepath);
        g_print("Creation d'un exemple de configuration...\n");

        if (!save_default_config(filepath))
        {
            g_printerr("Erreur: Impossible de creer le fichier de configuration\n");
            return false;
        }

        g_print("Veuillez editer %s et configurer vos identifiants SFTP\n", filepath);
        return false;
    }

    char line[512];
    bool in_sftp_section = false;
    bool found_host = false, found_user = false, found_pass = false;

    while (fgets(line, sizeof(line), fp))
    {
        char *p = g_strstrip(line);

        // Ignorer les commentaires et lignes vides
        if (*p == '#' || *p == '\n' || *p == '\0')
            continue;

        // Verifier section [sftp]
        if (strncmp(p, "[sftp]", 6) == 0)
        {
            in_sftp_section = true;
            continue;
        }

        // Nouvelle section
        if (*p == '[')
        {
            in_sftp_section = false;
            continue;
        }

        if (in_sftp_section)
        {
            gchar **parts = g_strsplit(p, "=", 2);
            if (g_strv_length(parts) == 2)
            {
                gchar *key = g_strstrip(parts[0]);
                gchar *value = g_strstrip(parts[1]);

                if (strcmp(key, "host") == 0)
                {
                    // Le host est obligatoire dans le fichier de config
                    strncpy(config->host, value, sizeof(config->host) - 1);
                    found_host = true;
                }
                else if (strcmp(key, "user") == 0)
                {
                    // Le user est obligatoire dans le fichier de config
                    strncpy(config->user, value, sizeof(config->user) - 1);
                    found_user = true;
                }
                else if (strcmp(key, "pass") == 0)
                {
                    // Le mot de passe est optionnel dans le fichier de config
                    // Il sera demandé à l'exécution si absent
                    strncpy(config->pass, value, sizeof(config->pass) - 1);
                    config->pass_provided = (strlen(value) > 0);
                }
            }
            g_strfreev(parts);
        }
    }

    fclose(fp);

    // if (!found_host || !found_user || !found_pass) {
    if (!found_host || !found_user)
    {
        g_printerr("Erreur: Configuration incomplete dans %s\n", filepath);
        g_printerr("Veuillez verifier les sections [sftp] host=, user=\n");
        return false;
    }

    // Verifier que les valeurs ne sont pas vides
    if (strlen(config->host) == 0 || strlen(config->user) == 0)
    {
        g_printerr("Erreur: Identifiants SFTP manquants dans %s\n", filepath);
        g_printerr("Veuillez configurer les sections [sftp] host=, user=\n");
        return false;
    }

    return true;
}

/**
 * @brief Sauvegarde un exemple de configuration
 */
static bool save_default_config(const char *filepath)
{
    GError *error = NULL;
    gchar *content = g_strdup_printf(
        "# Configuration Download Files\n"
        "[sftp]\n"
        "host = votre-serveur-sftp.com\n"
        "user = \n");

    gboolean success = g_file_set_contents(filepath, content, -1, &error);
    g_free(content);

    if (!success)
    {
        g_printerr("Erreur lors de la creation du fichier: %s\n", error->message);
        g_error_free(error);
        return false;
    }

    // Restreindre les permissions (600)
    chmod(filepath, S_IRUSR | S_IWUSR);

    return true;
}

/**
 * @brief Detecte automatiquement l'OS courant et retourne le nom (centos6, centos9, etc.)
 * @return Un string alloué dynamiquement avec le nom de l'OS, ou NULL si détection impossible
 */
static char *detect_osname(void)
{
    char *osname = NULL;
    FILE *fp;
    char line[256];
    int major_version = 0;

    // Essayer /etc/os-release (CentOS 7+, RHEL 7+, etc.)
    fp = fopen("/etc/os-release", "r");
    if (fp)
    {
        bool is_centos = false;
        bool is_rhel = false;

        while (fgets(line, sizeof(line), fp))
        {
            // Chercher ID=
            if (g_str_has_prefix(line, "ID="))
            {
                if (strstr(line, "centos") || strstr(line, "\"centos\""))
                {
                    is_centos = true;
                }
                else if (strstr(line, "rhel") || strstr(line, "\"rhel\""))
                {
                    is_rhel = true;
                }
            }

            // Chercher VERSION_ID=
            if (g_str_has_prefix(line, "VERSION_ID="))
            {
                // Extraire le numéro de version
                char *p = strchr(line, '=');
                if (p)
                {
                    p++; // Skip '='
                    // Enlever les guillemets
                    while (*p == '"' || *p == '\'')
                        p++;
                    major_version = atoi(p);
                }
            }
        }
        fclose(fp);

        if ((is_centos || is_rhel) && major_version > 0)
        {
            if (opt_verbose)
                g_print("Detection OS: CentOS/RHEL version %d\n", major_version);
            osname = g_strdup_printf("centos%d", major_version);
            return osname;
        }
    }

    // Essayer /etc/redhat-release ou /etc/centos-release (CentOS 6 et anciennes versions)
    const char *release_files[] = {
        "/etc/centos-release",
        "/etc/redhat-release",
        "/etc/system-release",
        NULL};

    for (int i = 0; release_files[i] != NULL; i++)
    {
        fp = fopen(release_files[i], "r");
        if (fp)
        {
            if (fgets(line, sizeof(line), fp))
            {
                // Parser des lignes comme:
                // "CentOS release 6.10 (Final)"
                // "Red Hat Enterprise Linux Server release 6.5 (Santiago)"
                // "CentOS Linux release 7.9.2009 (Core)"

                if (strstr(line, "CentOS") || strstr(line, "Red Hat"))
                {
                    // Chercher "release X.Y"
                    char *release_pos = strstr(line, "release ");
                    if (release_pos)
                    {
                        release_pos += 8; // Skip "release "
                        major_version = atoi(release_pos);

                        if (major_version > 0)
                        {
                            if (opt_verbose)
                                g_print("Detection OS: CentOS/RHEL version %d (via %s)\n",
                                        major_version, release_files[i]);
                            fclose(fp);
                            osname = g_strdup_printf("centos%d", major_version);
                            return osname;
                        }
                    }
                }
            }
            fclose(fp);
        }
    }

    // Si aucune détection n'a fonctionné
    if (opt_verbose)
        g_print("Detection OS: Impossible de determiner la version, utilisation de centos6 par defaut\n");

    return g_strdup("centos6");
}

/**
 * @brief Remplace le template {osname} dans une chaîne
 * @param template_str Chaîne contenant potentiellement {osname}
 * @param osname Valeur à substituer
 * @return Nouvelle chaîne allouée (doit être libérée avec g_free)
 */
static char *replace_template(const char *template_str, const char *osname)
{
    const char *placeholder = "{osname}";
    const char *pos = strstr(template_str, placeholder);

    // Pas de template, retourner une copie
    if (!pos)
    {
        return g_strdup(template_str);
    }

    // Calculer la taille nécessaire
    size_t prefix_len = pos - template_str;
    size_t suffix_len = strlen(pos + strlen(placeholder));
    size_t osname_len = strlen(osname);
    size_t total_len = prefix_len + osname_len + suffix_len + 1;

    // Allouer et construire la nouvelle chaîne
    char *result = g_malloc(total_len);

    // Copier prefix
    strncpy(result, template_str, prefix_len);
    result[prefix_len] = '\0';

    // Ajouter osname
    strcat(result, osname);

    // Ajouter suffix
    strcat(result, pos + strlen(placeholder));

    return result;
}

/**
 * @brief Lit un mot de passe depuis stdin sans l'afficher
 */
static char *read_password(const char *prompt)
{
    struct termios old_term, new_term;
    static char password[MAX_PASS];

    g_print("%s", prompt);
    fflush(stdout);

    // Desactiver l'echo terminal
    tcgetattr(STDIN_FILENO, &old_term);
    new_term = old_term;
    new_term.c_lflag &= ~(ECHO);
    tcsetattr(STDIN_FILENO, TCSANOW, &new_term);

    // Lire le mot de passe
    if (fgets(password, sizeof(password), stdin) == NULL)
    {
        tcsetattr(STDIN_FILENO, TCSANOW, &old_term);
        g_print("\n");
        return NULL;
    }

    // Restaurer l'echo terminal
    tcsetattr(STDIN_FILENO, TCSANOW, &old_term);
    g_print("\n");

    // Supprimer le retour à la ligne
    g_strchomp(password);

    return password;
}

/**
 * @brief Callback pour ecrire les donnees telechargees
 */
static size_t write_callback(void *ptr, size_t size, size_t nmemb, FILE *stream)
{
    return fwrite(ptr, size, nmemb, stream);
}

/**
 * @brief Callback pour lire les donnees a uploader
 */
static size_t read_callback(void *ptr, size_t size, size_t nmemb, FILE *stream)
{
    return fread(ptr, size, nmemb, stream);
}

/**
 * @brief Verifie et cree un repertoire si necessaire
 */
static bool ensure_directory(const char *path)
{
    GError *error = NULL;
    GFile *dir = g_file_new_for_path(path);

    if (g_file_query_exists(dir, NULL))
    {
        GFileInfo *info = g_file_query_info(dir, G_FILE_ATTRIBUTE_STANDARD_TYPE,
                                            G_FILE_QUERY_INFO_NONE, NULL, &error);
        if (error)
        {
            g_printerr("Erreur lors de la verification du repertoire: %s\n", error->message);
            g_error_free(error);
            g_object_unref(dir);
            return false;
        }

        GFileType file_type = g_file_info_get_file_type(info);
        g_object_unref(info);
        g_object_unref(dir);

        if (file_type != G_FILE_TYPE_DIRECTORY)
        {
            g_printerr("Erreur: %s existe mais n'est pas un repertoire\n", path);
            return false;
        }
        return true;
    }

    // Creer le repertoire avec mkdir -p
    if (!g_file_make_directory_with_parents(dir, NULL, &error))
    {
        g_printerr("Erreur: Impossible de creer le repertoire %s: %s\n", path, error->message);
        g_error_free(error);
        g_object_unref(dir);
        return false;
    }

    g_object_unref(dir);
    return true;
}

/**
 * @brief Telecharge un fichier via SFTP
 */
static bool download_file(const Config *config, const char *remote_path,
                          const char *local_path, const char *filename)
{
    CURL *curl;
    CURLcode res;
    FILE *fp;
    char url[MAX_PATH * 2];
    char local_file[MAX_PATH];
    char host_clean[MAX_HOST];

    // Nettoyer le hostname (enlever https:// ou http://)
    const char *h = config->host;
    if (g_str_has_prefix(h, "https://"))
        h += 8;
    else if (g_str_has_prefix(h, "http://"))
        h += 7;
    strncpy(host_clean, h, sizeof(host_clean) - 1);

    // Enlever le trailing slash ou path
    char *slash = strchr(host_clean, '/');
    if (slash)
        *slash = '\0';

    // Construire l'URL FTP
    snprintf(url, sizeof(url), "ftp://%s/%s%s",
             host_clean, remote_path, filename);

    // printf("URL: %s\n", url);

    // Construire le chemin local
    snprintf(local_file, sizeof(local_file), "%s/%s", local_path, filename);

    if (opt_verbose)
        g_print("Telechargement de %s depuis %s...\n", filename, url);
    else
        g_print("Telechargement de %s...\n", filename);

    // Ouvrir le fichier en ecriture
    fp = fopen(local_file, "wb");
    if (!fp)
    {
        g_printerr("Erreur: Impossible de creer %s: %s\n", local_file, strerror(errno));
        return false;
    }

    // Initialiser curl
    curl = curl_easy_init();
    if (!curl)
    {
        g_printerr("Erreur: Impossible d'initialiser curl\n");
        fclose(fp);
        return false;
    }

    // Configurer curl pour FTP
    curl_easy_setopt(curl, CURLOPT_URL, url);
    curl_easy_setopt(curl, CURLOPT_USERNAME, config->user);
    curl_easy_setopt(curl, CURLOPT_PASSWORD, config->pass);
    curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, write_callback);
    curl_easy_setopt(curl, CURLOPT_WRITEDATA, fp);
    curl_easy_setopt(curl, CURLOPT_VERBOSE, opt_verbose ? 1L : 0L);

    // Activer la barre de progression
    curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0L);
    curl_easy_setopt(curl, CURLOPT_XFERINFOFUNCTION, progress_callback);

    // Executer le telechargement
    res = curl_easy_perform(curl);

    // Afficher une nouvelle ligne apres la barre de progression
    printf("\n");

    // Nettoyer
    curl_easy_cleanup(curl);
    fclose(fp);

    if (res != CURLE_OK)
    {
        g_printerr("Erreur lors du telechargement de %s: %s\n",
                   filename, curl_easy_strerror(res));
        unlink(local_file); // Supprimer le fichier partiel
        return false;
    }

    // g_print("%s telechargement OK\n", filename);
    return true;
}

/**
 * @brief Calcule le hash MD5 d'un fichier
 * @param filename Chemin du fichier à hasher
 * @param output_hash Buffer pour stocker le hash (doit être au moins 33 caractères)
 * @return true si succès, false sinon
 */
static bool compute_md5(const char *filename, char *output_hash)
{
    FILE *fp = fopen(filename, "rb");
    if (!fp)
    {
        g_printerr("Erreur: Impossible d'ouvrir %s pour calcul MD5: %s\n",
                   filename, strerror(errno));
        return false;
    }

    MD5_CTX md5_ctx;
    MD5_Init(&md5_ctx);

    unsigned char buffer[8192];
    size_t bytes_read;

    while ((bytes_read = fread(buffer, 1, sizeof(buffer), fp)) > 0)
    {
        MD5_Update(&md5_ctx, buffer, bytes_read);
    }

    unsigned char digest[MD5_DIGEST_LENGTH];
    MD5_Final(digest, &md5_ctx);

    fclose(fp);

    // Convertir en chaîne hexadécimale
    for (int i = 0; i < MD5_DIGEST_LENGTH; i++)
    {
        sprintf(output_hash + (i * 2), "%02x", digest[i]);
    }
    output_hash[MD5_DIGEST_LENGTH * 2] = '\0';

    return true;
}

/**
 * @brief Vérifie le MD5 d'un fichier contre un hash attendu
 * @param filepath Chemin du fichier à vérifier
 * @param expected_md5 Hash MD5 attendu (format hexadécimal)
 * @return true si le hash correspond, false sinon
 */
static bool verify_md5(const char *filepath, const char *expected_md5)
{
    char computed_md5[MD5_DIGEST_LENGTH * 2 + 1];

    if (!compute_md5(filepath, computed_md5))
    {
        return false;
    }

    // Comparer les hash (insensible à la casse)
    if (g_ascii_strcasecmp(computed_md5, expected_md5) != 0)
    {
        g_printerr("ERREUR: Verification MD5 echouee pour %s\n", filepath);
        g_printerr("  Attendu : %s\n", expected_md5);
        g_printerr("  Calcule : %s\n", computed_md5);
        return false;
    }

    if (opt_verbose)
        g_print("Verification MD5 OK: %s\n", filepath);

    return true;
}

/**
 * @brief Télécharge un fichier et son MD5, puis vérifie l'intégrité
 * @param config Configuration FTP
 * @param remote_path Chemin distant
 * @param local_path Chemin local de destination
 * @param filename Nom du fichier à télécharger
 * @return true si succès et vérification OK, false sinon
 */
static bool download_and_verify_file(const Config *config, const char *remote_path,
                                     const char *local_path, const char *filename)
{
    char local_filepath[MAX_PATH];
    char md5_filename[MAX_FILENAME + 5];
    char local_md5_path[MAX_PATH];

    // Télécharger le fichier principal
    if (!download_file(config, remote_path, local_path, filename))
    {
        return false;
    }

    // Construire les chemins pour le fichier MD5
    snprintf(md5_filename, sizeof(md5_filename), "%s.md5", filename);
    snprintf(local_filepath, sizeof(local_filepath), "%s/%s", local_path, filename);
    snprintf(local_md5_path, sizeof(local_md5_path), "%s/%s", local_path, md5_filename);

    // Télécharger le fichier MD5
    if (opt_verbose)
        g_print("Telechargement du fichier MD5: %s\n", md5_filename);

    if (!download_file(config, remote_path, local_path, md5_filename))
    {
        g_print("Avertissement: Fichier MD5 non disponible pour %s, verification ignoree\n",
                filename);
        return true; // Continuer même si pas de MD5
    }

    // Lire le hash MD5 attendu depuis le fichier
    FILE *md5_fp = fopen(local_md5_path, "r");
    if (!md5_fp)
    {
        g_printerr("Erreur: Impossible de lire %s: %s\n",
                   local_md5_path, strerror(errno));
        return false;
    }

    char expected_md5[MD5_DIGEST_LENGTH * 2 + 256]; // Buffer large pour le format md5sum
    if (fgets(expected_md5, sizeof(expected_md5), md5_fp) == NULL)
    {
        g_printerr("Erreur: Fichier MD5 vide: %s\n", local_md5_path);
        fclose(md5_fp);
        return false;
    }
    fclose(md5_fp);

    // Parser le format md5sum: "hash  filename" ou juste "hash"
    char *space = strchr(expected_md5, ' ');
    if (space)
    {
        *space = '\0'; // Tronquer au premier espace
    }
    g_strchomp(expected_md5); // Enlever les espaces/newlines trailing

    // Vérifier le MD5
    // g_print("Verification de l'integrite de %s...  ", filename);
    g_print("Verification de l'integrite... ");
    if (!verify_md5(local_filepath, expected_md5))
    {
        g_printerr("ECHEC: Fichier corrompu ou incomplet: %s\n", filename);
        g_printerr("Suppression du fichier invalide...\n");
        unlink(local_filepath);
        unlink(local_md5_path);
        return false;
    }

    g_print("OK\n\n");

    // Supprimer le fichier .md5 après vérification réussie (optionnel)
    if (!opt_verbose)
        unlink(local_md5_path);

    return true;
}

/**
 * @brief Upload un fichier via FTP
 */
static bool upload_file(const Config *config, const char *remote_path,
                        const char *local_file)
{
    CURL *curl;
    CURLcode res;
    FILE *fp;
    char url[MAX_PATH * 2];
    char host_clean[MAX_HOST];
    struct stat file_info;

    // Verifier que le fichier local existe
    if (stat(local_file, &file_info) != 0)
    {
        g_printerr("Erreur: Fichier local %s introuvable: %s\n", local_file, strerror(errno));
        return false;
    }

    // Extraire le nom du fichier depuis le chemin complet
    const char *filename = strrchr(local_file, '/');
    if (filename)
    {
        filename++; // Skip le '/'
    }
    else
    {
        filename = local_file;
    }

    // Nettoyer le hostname (enlever https:// ou http://)
    const char *h = config->host;
    if (g_str_has_prefix(h, "https://"))
        h += 8;
    else if (g_str_has_prefix(h, "http://"))
        h += 7;
    strncpy(host_clean, h, sizeof(host_clean) - 1);

    // Enlever le trailing slash ou path
    char *slash = strchr(host_clean, '/');
    if (slash)
        *slash = '\0';

    // Construire l'URL FTP
    snprintf(url, sizeof(url), "ftp://%s/%s%s",
             host_clean, remote_path, filename);

    if (opt_verbose)
        g_print("Upload de %s vers %s...\n", local_file, url);
    else
        g_print("Upload de %s...\n", filename);

    // Ouvrir le fichier en lecture
    fp = fopen(local_file, "rb");
    if (!fp)
    {
        g_printerr("Erreur: Impossible d'ouvrir %s: %s\n", local_file, strerror(errno));
        return false;
    }

    // Initialiser curl
    curl = curl_easy_init();
    if (!curl)
    {
        g_printerr("Erreur: Impossible d'initialiser curl\n");
        fclose(fp);
        return false;
    }

    // Configurer curl pour FTP upload
    curl_easy_setopt(curl, CURLOPT_URL, url);
    curl_easy_setopt(curl, CURLOPT_USERNAME, config->user);
    curl_easy_setopt(curl, CURLOPT_PASSWORD, config->pass);
    curl_easy_setopt(curl, CURLOPT_UPLOAD, 1L);
    curl_easy_setopt(curl, CURLOPT_READFUNCTION, read_callback);
    curl_easy_setopt(curl, CURLOPT_READDATA, fp);
    curl_easy_setopt(curl, CURLOPT_INFILESIZE_LARGE, (curl_off_t)file_info.st_size);
    curl_easy_setopt(curl, CURLOPT_VERBOSE, opt_verbose ? 1L : 0L);

    // Activer la barre de progression pour upload
    curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0L);
    curl_easy_setopt(curl, CURLOPT_XFERINFOFUNCTION, progress_callback);

    // Executer l'upload
    res = curl_easy_perform(curl);

    // Afficher une nouvelle ligne apres la barre de progression
    printf("\n");

    // Nettoyer
    curl_easy_cleanup(curl);
    fclose(fp);

    if (res != CURLE_OK)
    {
        g_printerr("Erreur lors de l'upload de %s: %s\n",
                   filename, curl_easy_strerror(res));
        return false;
    }

    g_print("%s upload OK\n", filename);
    return true;
}

/**
 * @brief Cree une liste de fichiers
 */
static FileList *create_file_list(void)
{
    FileList *list = g_malloc(sizeof(FileList));
    list->capacity = 100;
    list->count = 0;
    list->filenames = g_malloc(sizeof(char *) * list->capacity);
    return list;
}

/**
 * @brief Ajoute un fichier à la liste
 */
static void add_file_to_list(FileList *list, const char *filename)
{
    if (list->count >= list->capacity)
    {
        list->capacity *= 2;
        list->filenames = g_realloc(list->filenames, sizeof(char *) * list->capacity);
    }

    list->filenames[list->count] = g_strdup(filename);
    list->count++;
}

/**
 * @brief Libère la liste de fichiers
 */
static void free_file_list(FileList *list)
{
    if (!list)
        return;

    for (int i = 0; i < list->count; i++)
    {
        g_free(list->filenames[i]);
    }
    g_free(list->filenames);
    g_free(list);
}

/**
 * @brief Callback pour lire le listing de repertoire
 */
static size_t list_callback(void *ptr, size_t size, size_t nmemb, void *userdata)
{
    FileList *list = (FileList *)userdata;
    size_t total_size = size * nmemb;
    char *data = (char *)ptr;

    // Parser la sortie ligne par ligne
    // Avec CURLOPT_DIRLISTONLY, chaque ligne est un nom de fichier
    gchar **lines = g_strsplit(data, "\n", -1);
    for (int i = 0; lines[i] != NULL; i++)
    {
        gchar *line = g_strstrip(lines[i]);
        if (strlen(line) > 0 && strcmp(line, ".") != 0 && strcmp(line, "..") != 0)
        {
            add_file_to_list(list, line);
        }
    }
    g_strfreev(lines);

    return total_size;
}

/**
 * @brief Liste les fichiers distants
 */
static bool list_remote_files(const Config *config, const char *remote_path, FileList *list)
{
    CURL *curl;
    CURLcode res;
    char url[MAX_PATH * 2];
    char host_clean[MAX_HOST];

    // // Nettoyer le hostname
    // const char *h = config->host;
    // if (g_str_has_prefix(h, "https://")) h += 8;
    // else if (g_str_has_prefix(h, "http://")) h += 7;
    // strncpy(host_clean, h, sizeof(host_clean) - 1);

    char *slash = strchr(config->host, '/');
    if (slash)
        *slash = '\0';

    // Construire l'URL pour FTP
    snprintf(url, sizeof(url), "ftp://%s/%s", config->host, remote_path);

    curl = curl_easy_init();
    if (!curl)
    {
        g_printerr("Erreur: Impossible d'initialiser curl\n");
        return false;
    }

    curl_easy_setopt(curl, CURLOPT_URL, url);
    curl_easy_setopt(curl, CURLOPT_USERNAME, config->user);
    curl_easy_setopt(curl, CURLOPT_PASSWORD, config->pass);
    curl_easy_setopt(curl, CURLOPT_DIRLISTONLY, 1L);
    curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, list_callback);
    curl_easy_setopt(curl, CURLOPT_WRITEDATA, list);

    res = curl_easy_perform(curl);
    curl_easy_cleanup(curl);

    if (res != CURLE_OK)
    {
        g_printerr("Erreur lors du listage: %s\n", curl_easy_strerror(res));
        return false;
    }

    return true;
}

/**
 * @brief Trouve la dernière archive GCV
 */
static char *find_latest_gc_archive(FileList *list, const char *osname, char *version_out)
{
    regex_t regex;
    char pattern[256];
    char *latest_file = NULL;
    char latest_version[MAX_VERSION] = "";

    // Pattern depend de l'OS: Gestion-osname-VERSION.tar.*
    snprintf(pattern, sizeof(pattern),
             "^Gestion-%s-([0-9]+\\.[0-9]+\\.[0-9]+[a-z]+.*)\\.tar\\.(bz2|xz)$",
             osname);

    if (regcomp(&regex, pattern, REG_EXTENDED) != 0)
    {
        g_printerr("Erreur: Impossible de compiler l'expression regulière\n");
        return NULL;
    }

    regmatch_t matches[3];

    for (int i = 0; i < list->count; i++)
    {
        if (regexec(&regex, list->filenames[i], 3, matches, 0) == 0)
        {
            // Extraire la version
            int start = matches[1].rm_so;
            int end = matches[1].rm_eo;
            char version[MAX_VERSION];
            int len = end - start;
            if (len >= MAX_VERSION)
                len = MAX_VERSION - 1;
            strncpy(version, list->filenames[i] + start, len);
            version[len] = '\0';

            // Comparer les versions (simple comparaison lexicographique)
            if (strcmp(version, latest_version) > 0)
            {
                strcpy(latest_version, version);
                latest_file = list->filenames[i];
            }
        }
    }

    regfree(&regex);

    if (latest_file && version_out)
    {
        strcpy(version_out, latest_version);
    }

    return latest_file;
}

static bool load_config_then_ask_password(Config *config)
{
    // Charger la configuration
    if (!load_config(config))
    {
        return false;
    }

    if (config->pass_provided)
    {
        return true; // Le mot de passe est déjà dans la config
    }

    // Lire le mot de passe depuis le terminal
    char *pass = read_password("Mot de passe SFTP: ");
    if (!pass || strlen(pass) == 0)
    {
        g_printerr("Erreur: Impossible de lire le mot de passe\n");
        return false;
    }
    strncpy(config->pass, pass, sizeof(config->pass) - 1);
    memset(pass, 0, strlen(pass)); // Effacer le mot de passe de la m�moire

    return true;
}

/**
 * @brief Commande download
 */
static int cmd_download(const char *osname, const char *local_dir, bool list_only)
{
    Config config = {0};

    if (!load_config_then_ask_password(&config))
    {
        return 1;
    }

    // Crer le rpertoire local
    if (!ensure_directory(local_dir))
    {
        return 1;
    }

    // Construire le chemin distant
    gchar *remote_path = g_strdup_printf("%s/", osname);

    // Lister les fichiers distants
    FileList *remote_files = create_file_list();
    if (!remote_files)
    {
        g_printerr("Erreur: Allocation memoire\n");
        g_free(remote_path);
        return 1;
    }

    g_print("Connexion au serveur SFTP...\n");
    if (!list_remote_files(&config, remote_path, remote_files))
    {
        free_file_list(remote_files);
        g_free(remote_path);
        return 1;
    }

    if (list_only)
    {
        g_print("Fichiers disponibles dans %s:\n", remote_path);
        for (int i = 0; i < remote_files->count; i++)
        {
            g_print("  %s\n", remote_files->filenames[i]);
        }
        free_file_list(remote_files);
        g_free(remote_path);
        return 0;
    }

    // =========================================================================
    // Telecharger les fichiers relatifs à l'OS (liste FILES_TO_DOWNLOAD)
    // =========================================================================
    bool all_ok = true;

    if (NUM_FILES_TO_DOWNLOAD > 0)
    {
        // g_print("\n=== Telechargement des fichiers relatifs a %s ===\n", osname);

        for (int i = 0; i < NUM_FILES_TO_DOWNLOAD; i++)
        {
            // Remplacer {osname} dans le nom de fichier si présent
            char *filename = replace_template(FILES_TO_DOWNLOAD[i], osname);

            bool found = false;
            for (int j = 0; j < remote_files->count; j++)
            {
                if (strcmp(filename, remote_files->filenames[j]) == 0)
                {
                    found = true;
                    break;
                }
            }

            if (found)
            {
                if (!download_and_verify_file(&config, remote_path, local_dir, filename))
                {
                    all_ok = false;
                }
            }
            else
            {
                g_print("Fichier %s non trouve sur le serveur\n", filename);
            }

            g_free(filename);
        }
    }

    // Libérer les ressources pour les fichiers relatifs
    free_file_list(remote_files);
    g_free(remote_path);

    // =========================================================================
    // Telecharger les fichiers avec chemins absolus (liste FILES_ABSOLUTE_PATHS)
    // =========================================================================
    if (NUM_FILES_ABSOLUTE > 0)
    {
        // g_print("\n=== Telechargement des fichiers avec chemins absolus ===\n");

        for (int i = 0; i < NUM_FILES_ABSOLUTE; i++)
        {
            // Extraire le nom du fichier depuis le chemin complet
            const char *full_path = FILES_ABSOLUTE_PATHS[i];
            if (full_path[0] != '/')
            {
                g_printerr("Erreur: Le chemin %s n'est pas absolu\n", full_path);
                all_ok = false;
                continue;
            }

            full_path++; // Skip le '/'
            char filename[MAX_FILENAME];
            // // Extraire le répertoire distant et s'assurer qu'il commence par '/'
            char remote_dir[MAX_PATH];
            const char *last_slash = strrchr(full_path, '/');
            if (last_slash)
            {
                size_t dir_len = (last_slash + 1) - full_path;
                strncpy(remote_dir, full_path, dir_len);
                printf("DIR LEN: %zu\n", dir_len);
                printf("remote_dir before null: %s\n", remote_dir);
                remote_dir[dir_len] = '\0';

                strcpy(filename, last_slash + 1);
            }
            else
            {
                strcpy(remote_dir, "");
                strcpy(filename, full_path);
            }

            g_print("\nTelechargement de %s depuis %s...\n", filename, remote_dir);
            if (!download_and_verify_file(&config, remote_dir, local_dir, filename))
            {
                all_ok = false;
            }
        }
    }

    if (all_ok)
    {
        g_print("\nTelechargement termine\n");
        return 0;
    }

    return 1;
}

/**
 * @brief Commande latest
 */
static int cmd_latest(const char *osname, const char *local_dir, bool version_only)
{
    Config config = {0};

    if (!load_config_then_ask_password(&config))
    {
        return 1;
    }

    if (!ensure_directory(local_dir))
    {
        return 1;
    }

    gchar *remote_path = g_strdup_printf("%s/", osname);

    FileList *remote_files = create_file_list();
    if (!remote_files)
    {
        g_printerr("Erreur: Allocation memoire\n");
        g_free(remote_path);
        return 1;
    }

    g_print("Connexion au serveur SFTP...\n");
    if (!list_remote_files(&config, remote_path, remote_files))
    {
        free_file_list(remote_files);
        g_free(remote_path);
        return 1;
    }

    char version[MAX_VERSION];
    char *gc_archive = find_latest_gc_archive(remote_files, osname, version);

    if (!gc_archive)
    {
        g_printerr("Aucune version trouvee dans le repertoire distant.\n");
        free_file_list(remote_files);
        g_free(remote_path);
        return 1;
    }

    if (version_only)
    {
        g_print("%s\n", version);
        free_file_list(remote_files);
        g_free(remote_path);
        return 0;
    }

    g_print("=======\n");
    g_print("= Dernière version trouvée : %s (version: %s)\n", gc_archive, version);
    g_print("=======\n");

    bool result = download_and_verify_file(&config, remote_path, local_dir, gc_archive);

    free_file_list(remote_files);
    g_free(remote_path);

    if (result)
    {
        // g_print("\nArchive %s telechargee avec succes\n", gc_archive);
        return 0;
    }

    return 1;
}

/**
 * @brief Commande down file
 */
static int cmd_down(const char *osname, const char *local_dir, int num_files, char **files)
{
    Config config = {0};

    if (!load_config_then_ask_password(&config))
    {
        return 1;
    }

    if (!ensure_directory(local_dir))
    {
        return 1;
    }

    gchar *remote_path = g_strdup_printf("%s/", osname);

    bool all_ok = true;
    short i;
    for (i = 0; i < num_files; i++)
    {
        bool result = download_and_verify_file(&config, remote_path, local_dir, files[i]);
        all_ok = all_ok && result;
    }

    g_free(remote_path);

    if (all_ok)
    {
        g_print("\nUpload termine avec succes\n");
        return 0;
    }

    return 1;
}

/**
 * @brief Commande version
 */
static int cmd_version(const char *version_arg, const char *osname, const char *local_dir)
{
    Config config = {0};

    if (!load_config_then_ask_password(&config))
    {
        return 1;
    }

    if (!ensure_directory(local_dir))
    {
        return 1;
    }

    gchar *remote_path = g_strdup_printf("%s/", osname);

    FileList *remote_files = create_file_list();
    if (!remote_files)
    {
        g_printerr("Erreur: Allocation memoire\n");
        g_free(remote_path);
        return 1;
    }

    g_print("Connexion au serveur SFTP...\n");
    if (!list_remote_files(&config, remote_path, remote_files))
    {
        free_file_list(remote_files);
        g_free(remote_path);
        return 1;
    }

    // Chercher le fichier avec cette version
    gchar *possible_files[4];
    possible_files[0] = g_strdup_printf("Gestion-%s-%s.tar.bz2", osname, version_arg);
    possible_files[1] = g_strdup_printf("Gestion-%s-%s.tar.xz", osname, version_arg);
    possible_files[2] = NULL;
    possible_files[3] = NULL;

    char *gc_archive = NULL;
    for (int i = 0; i < 2 && possible_files[i] != NULL; i++)
    {
        for (int j = 0; j < remote_files->count; j++)
        {
            if (strcmp(possible_files[i], remote_files->filenames[j]) == 0)
            {
                gc_archive = remote_files->filenames[j];
                break;
            }
        }
        if (gc_archive)
            break;
    }

    if (!gc_archive)
    {
        g_printerr("Version %s non trouvee sur le serveur\n", version_arg);
        g_printerr("Fichiers recherches :\n");
        for (int i = 0; i < 4 && possible_files[i] != NULL; i++)
        {
            g_printerr("  - %s\n", possible_files[i]);
        }
        for (int i = 0; i < 4 && possible_files[i] != NULL; i++)
        {
            g_free(possible_files[i]);
        }
        free_file_list(remote_files);
        g_free(remote_path);
        return 1;
    }

    g_print("=======\n");
    g_print("= Version trouvee : %s\n", gc_archive);
    g_print("=======\n");

    bool result = download_and_verify_file(&config, remote_path, local_dir, gc_archive);

    for (int i = 0; i < 4 && possible_files[i] != NULL; i++)
    {
        g_free(possible_files[i]);
    }
    free_file_list(remote_files);
    g_free(remote_path);

    if (result)
    {
        // g_print("\nArchive %s telechargee avec succes\n", gc_archive);
        return 0;
    }

    return 1;
}

/**
 * @brief Commande upload
 */
static int cmd_upload(const char *osname, int num_files, char **files)
{
    Config config = {0};

    if (!load_config_then_ask_password(&config))
    {
        return 1;
    }

    // Construire le chemin distant
    gchar *remote_path = g_strdup_printf("%s/", osname);

    g_print("Upload vers %s sur le serveur FTP...\n", remote_path);

    // Uploader chaque fichier
    bool all_ok = true;
    for (int i = 0; i < num_files; i++)
    {
        if (!upload_file(&config, remote_path, files[i]))
        {
            all_ok = false;
        }
    }

    g_free(remote_path);

    if (all_ok)
    {
        g_print("\nUpload termine avec succes\n");
        return 0;
    }

    return 1;
}

/**
 * @brief Commande check - Verifie la connexion au serveur SFTP
 */
static int cmd_check(const char *osname)
{
    Config config = {0};

    if (!load_config_then_ask_password(&config))
    {
        return 1;
    }

    g_print("=======\n");
    g_print("= Verification de la connexion au serveur SFTP\n");
    g_print("=======\n");

    // Tester la connexion en listant les fichiers distants
    gchar *remote_path = g_strdup_printf("%s/", osname);
    FileList *remote_files = create_file_list();

    if (!remote_files)
    {
        g_printerr("Erreur: Allocation memoire\n");
        g_free(remote_path);
        return 1;
    }

    g_print("Test de connexion...\n");
    if (!list_remote_files(&config, remote_path, remote_files))
    {
        g_printerr("\n[ECHEC] Impossible de se connecter au serveur SFTP\n");
        g_printerr("Verifiez:\n");
        g_printerr("  - L'adresse du serveur dans config.ini\n");
        g_printerr("  - Vos identifiants (user/pass)\n");
        g_printerr("  - Que le serveur est accessible (firewall, reseau)\n");
        g_printerr("  - Que le repertoire '%s' existe sur le serveur\n", osname);

        free_file_list(remote_files);
        g_free(remote_path);
        return 1;
    }

    g_print("\n[OK] Connexion reussie!\n");
    free_file_list(remote_files);
    g_free(remote_path);

    return 0;
}

/**
 * @brief Point d'entree principal
 */
int main(int argc, char *argv[])
{
    GError *error = NULL;
    GOptionContext *context;
    int result = 0;

    // Initialiser GLib type system (necessaire pour CentOS 6 / GLib < 2.36)
#if !GLIB_CHECK_VERSION(2, 36, 0)
    g_type_init();
#endif

    // Definir les options communes
    GOptionEntry common_entries[] = {
        {"osname", 'o', 0, G_OPTION_ARG_STRING, &opt_osname,
         "Nom du système d'exploitation (defaut: detection auto)", "OS"},
        {"dir", 'd', 0, G_OPTION_ARG_STRING, &opt_dir,
         "Repertoire local de destination (defaut: files)", "DIR"},
        {"config", 'c', 0, G_OPTION_ARG_STRING, &opt_config,
         "Chemin vers le fichier de configuration (defaut: ./config.ini)", "CONFIG"},
        {"verbose", 'V', 0, G_OPTION_ARG_NONE, &opt_verbose,
         "Mode verbeux", NULL},
        {NULL}};

    // Options pour la commande download
    GOptionEntry download_entries[] = {
        {"list", 'l', 0, G_OPTION_ARG_NONE, &opt_list,
         "Liste uniquement les fichiers disponibles", NULL},
        {NULL}};

    // Options pour la commande latest
    GOptionEntry latest_entries[] = {
        {"version-only", 'v', 0, G_OPTION_ARG_NONE, &opt_version_only,
         "Affiche seulement le nom de la version", NULL},
        {NULL}};

    // Creer le contexte principal
    context = g_option_context_new("COMMAND [OPTIONS] - Telechargement de fichiers GCV via SFTP");
    g_option_context_set_summary(context,
                                 "Commandes disponibles:\n"
                                 "  check          Verifie la connexion au serveur SFTP\n"
                                 "  download       Telecharge les fichiers de dependances\n"
                                 "  latest         Telecharge la dernière version de l'archive GCV\n"
                                 "  version VER    Telecharge une version specifique\n"
                                 "  upload FILES   Upload un ou plusieurs fichiers vers le serveur\n"
                                 "\n"
                                 "Configuration:\n"
                                 "  Le programme lit ./config.ini pour les identifiants SFTP (ou un fichier specifie via -c).\n"
                                 "  Si le fichier n'existe pas, il sera cree avec un exemple de configuration.");

    g_option_context_set_description(context,
                                     "Exemples:\n"
                                     "  download_files check                       # Verifie la connexion SFTP\n"
                                     "  download_files download                    # Telecharge les dependances\n"
                                     "  download_files latest -o centos9           # Dernière version pour CentOS 9\n"
                                     "  download_files version 6.06.011ad          # Version specifique\n"
                                     "  download_files download --list             # Liste les fichiers disponibles\n"
                                     "  download_files upload -o centos9 file.tar.xz  # Upload un fichier vers centos9/\n"
                                     "  download_files -c /path/to/config.ini latest  # Utiliser un fichier de config specifique");

    // Initialiser libcurl
    curl_global_init(CURL_GLOBAL_DEFAULT);

    // Verifier qu'il y a au moins une commande
    if (argc < 2)
    {
        gchar *help_text = g_option_context_get_help(context, TRUE, NULL);
        g_print("%s", help_text);
        g_free(help_text);
        g_option_context_free(context);
        curl_global_cleanup();
        return 1;
    }

    const char *command = argv[1];

    // Parser selon la commande
    if (strcmp(command, "check") == 0)
    {
        g_option_context_add_main_entries(context, common_entries, NULL);

        if (!g_option_context_parse(context, &argc, &argv, &error))
        {
            g_printerr("Erreur de parsing des options: %s\n", error->message);
            g_error_free(error);
            g_option_context_free(context);
            curl_global_cleanup();
            return 1;
        }

        char *detected_osname = NULL;
        const char *osname;
        if (opt_osname)
        {
            osname = opt_osname;
        }
        else
        {
            detected_osname = detect_osname();
            osname = detected_osname;
            g_print("OS detecte: %s\n", osname);
        }

        result = cmd_check(osname);

        if (detected_osname)
            g_free(detected_osname);
    }
    else if (strcmp(command, "download") == 0)
    {
        g_option_context_add_main_entries(context, common_entries, NULL);
        g_option_context_add_main_entries(context, download_entries, NULL);

        if (!g_option_context_parse(context, &argc, &argv, &error))
        {
            g_printerr("Erreur de parsing des options: %s\n", error->message);
            g_error_free(error);
            g_option_context_free(context);
            curl_global_cleanup();
            return 1;
        }

        char *detected_osname = NULL;
        const char *osname;
        if (opt_osname)
        {
            osname = opt_osname;
        }
        else
        {
            detected_osname = detect_osname();
            osname = detected_osname;
            g_print("OS detecte: %s\n", osname);
        }
        const char *local_dir = opt_dir ? opt_dir : "files";

        result = cmd_download(osname, local_dir, opt_list);

        if (detected_osname)
            g_free(detected_osname);
    }
    else if (strcmp(command, "latest") == 0)
    {
        g_option_context_add_main_entries(context, common_entries, NULL);
        g_option_context_add_main_entries(context, latest_entries, NULL);

        if (!g_option_context_parse(context, &argc, &argv, &error))
        {
            g_printerr("Erreur de parsing des options: %s\n", error->message);
            g_error_free(error);
            g_option_context_free(context);
            curl_global_cleanup();
            return 1;
        }

        char *detected_osname = NULL;
        const char *osname;
        if (opt_osname)
        {
            osname = opt_osname;
        }
        else
        {
            detected_osname = detect_osname();
            osname = detected_osname;
            g_print("OS detecte: %s\n", osname);
        }
        const char *local_dir = opt_dir ? opt_dir : "files";

        result = cmd_latest(osname, local_dir, opt_version_only);

        if (detected_osname)
            g_free(detected_osname);
    }
    else if (strcmp(command, "version") == 0)
    {
        g_option_context_add_main_entries(context, common_entries, NULL);

        if (!g_option_context_parse(context, &argc, &argv, &error))
        {
            g_printerr("Erreur de parsing des options: %s\n", error->message);
            g_error_free(error);
            g_option_context_free(context);
            curl_global_cleanup();
            return 1;
        }

        if (argc < 3)
        {
            g_printerr("Erreur: La commande 'version' necessite un argument VERSION\n");
            gchar *help_text = g_option_context_get_help(context, TRUE, NULL);
            g_print("%s", help_text);
            g_free(help_text);
            result = 1;
        }
        else
        {
            const char *version_arg = argv[2];
            char *detected_osname = NULL;
            const char *osname;
            if (opt_osname)
            {
                osname = opt_osname;
            }
            else
            {
                detected_osname = detect_osname();
                osname = detected_osname;
                g_print("OS detecte: %s\n", osname);
            }
            const char *local_dir = opt_dir ? opt_dir : "files";

            result = cmd_version(version_arg, osname, local_dir);

            if (detected_osname)
                g_free(detected_osname);
        }
    }
    else if (strcmp(command, "upload") == 0)
    {
        g_option_context_add_main_entries(context, common_entries, NULL);

        if (!g_option_context_parse(context, &argc, &argv, &error))
        {
            g_printerr("Erreur de parsing des options: %s\n", error->message);
            g_error_free(error);
            g_option_context_free(context);
            curl_global_cleanup();
            return 1;
        }

        if (argc < 3)
        {
            g_printerr("Erreur: La commande 'upload' necessite au moins un fichier\n");
            gchar *help_text = g_option_context_get_help(context, TRUE, NULL);
            g_print("%s", help_text);
            g_free(help_text);
            result = 1;
        }
        else
        {
            char *detected_osname = NULL;
            const char *osname;
            if (opt_osname)
            {
                osname = opt_osname;
            }
            else
            {
                detected_osname = detect_osname();
                osname = detected_osname;
                g_print("OS detecte: %s\n", osname);
            }
            int num_files = argc - 2;
            char **files = &argv[2];

            result = cmd_upload(osname, num_files, files);

            if (detected_osname)
                g_free(detected_osname);
        }
    }
    else if (strcmp(command, "down") == 0)
    {
        g_option_context_add_main_entries(context, common_entries, NULL);

        if (!g_option_context_parse(context, &argc, &argv, &error))
        {
            g_printerr("Erreur de parsing des options: %s\n", error->message);
            g_error_free(error);
            g_option_context_free(context);
            curl_global_cleanup();
            return 1;
        }

        if (argc < 3)
        {
            g_printerr("Erreur: La commande 'down' necessite au moins un fichier\n");
            gchar *help_text = g_option_context_get_help(context, TRUE, NULL);
            g_print("%s", help_text);
            g_free(help_text);
            result = 1;
        }
        else
        {
            char *detected_osname = NULL;
            const char *osname;
            if (opt_osname)
            {
                osname = opt_osname;
            }
            else
            {
                detected_osname = detect_osname();
                osname = detected_osname;
                g_print("OS detecte: %s\n", osname);
            }
            int num_files = argc - 2;
            char **files = &argv[2];
            const char *local_dir = opt_dir ? opt_dir : "files";

            result = cmd_down(osname, local_dir, num_files, files);

            if (detected_osname)
                g_free(detected_osname);
        }
    }
    else if (strcmp(command, "help") == 0 || strcmp(command, "--help") == 0 || strcmp(command, "-h") == 0)
    {
        gchar *help_text = g_option_context_get_help(context, TRUE, NULL);
        g_print("%s", help_text);
        g_free(help_text);
        result = 0;
    }
    else
    {
        g_printerr("Commande inconnue: %s\n\n", command);
        gchar *help_text = g_option_context_get_help(context, TRUE, NULL);
        g_print("%s", help_text);
        g_free(help_text);
        result = 1;
    }

    // Nettoyage
    g_free(opt_osname);
    g_free(opt_dir);
    g_free(opt_config);
    g_option_context_free(context);
    curl_global_cleanup();

    return result;
}

/**
 * @brief Callback de progression pour libcurl
 */
static int progress_callback(void *clientp,
                             curl_off_t dltotal, curl_off_t dlnow,
                             curl_off_t ultotal, curl_off_t ulnow)
{
    (void)clientp;
    (void)ultotal;
    (void)ulnow;

    if (dltotal <= 0)
        return 0;

    double progress = (double)dlnow / (double)dltotal;
    int bar_width = 40;
    int pos = (int)(bar_width * progress);

    printf("\r[");
    for (int i = 0; i < bar_width; ++i)
    {
        if (i < pos)
            printf("=");
        else if (i == pos)
            printf(">");
        else
            printf(" ");
    }
    printf("] %3d%%", (int)(progress * 100));
    fflush(stdout);

    return 0; // 0 = continuer, non-0 = annuler
}
