#!/bin/env python3
"""Module pyisql - Interface pour interroger la base de données Informix.

Ce module fournit des fonctions pour gérer les clients, articles, factures et configuration CRM.
"""

from typing import Tuple, Dict, List, Optional

# Import des sous-modules
from . import crm
from . import factures
from . import article

# Import des fonctions IfxPy
from .itfifxpy import (
    fetch_one,
    fetch_all,
    fetch_value,
    execute_query,
    execute_update,
    unload_to_file
)


def split_rsoc(rsoc: str) -> Optional[Dict[str, str]]:
    """Parse le champ rsoc (raison sociale / nom complet) en composants.

    Args:
        rsoc: Champ raison sociale contenant genre, nom et prénom

    Returns:
        Dictionnaire {"gender": str, "lastname": str, "firstname": str} ou None si RGPD

    Note:
        Certains rsoc ont perdu l'espace entre le genre et le nom, ce qui peut
        conduire à perdre la première lettre du nom de famille.
    """
    if rsoc.strip() == "FICHE SUPPRIMEE RGPD":
        return None

    # Extraction du genre
    gender = rsoc[:3].strip().lower()
    if gender not in ["mr", "mle", "mme"]:
        print(f"Genre inattendu dans rsoc: {rsoc}")

    # Reconstruction avec espaces pour compenser les rsoc mal formatés
    rsoc = "    " + rsoc[len(gender):].lstrip()

    lastname = rsoc[4:17].strip()
    firstname = rsoc[18:].strip()

    return {"gender": gender, "lastname": lastname, "firstname": firstname}


def select_crtf_cli(code: str, is_actif: bool = True, columns: Optional[List[str]] = None) -> Tuple[int, Dict]:
    """Sélectionne un client certifié par code (avec données de gc_cli et gc_cli2).

    Args:
        code: Code du client
        is_actif: Filtrer par statut actif (True/False/None pour tous)
        columns: Colonnes à récupérer de gc_cli (défaut: colonnes standard)

    Returns:
        Tuple (nbrows, client) où:
        - nbrows: 0 ou 1
        - client: Dictionnaire avec les données (vide si non trouvé)

    Raises:
        ValueError: Si plusieurs clients trouvés
    """
    if columns is None:
        columns = [
            "code", "rsoc", "ad1", "ad2", "cpv", "codpost",
            "tel", "cmpt", "crtf", "birth", "email", "activite"
        ]

    # Requête gc_cli
    sql = f"SELECT {','.join(columns)} FROM gc_cli WHERE code='{code}' AND crtf='1'"

    if is_actif is not None:
        activite = '1' if is_actif else '0'
        sql += f" AND activite='{activite}'"

    try:
        client = fetch_one(sql)
    except ValueError as e:
        raise ValueError(f"Plusieurs clients trouvés avec le code '{code}': {e}")

    if client is None:
        return 0, {}

    # Requête gc_cli2
    columns2 = ['ad_app', 'ad_bat', 'ad_num', 'ad_rue', 'ad_ldt', 'sgmnt', 'numcrt']
    sql2 = f"SELECT {','.join(columns2)} FROM gc_cli2 WHERE code='{code}' AND if_oubli='0'"

    try:
        client2 = fetch_one(sql2)
    except ValueError as e:
        raise ValueError(f"Plusieurs enregistrements cli2 trouvés pour '{code}': {e}")

    if client2 is None:
        return 0, {}

    # Fusionner les données
    client.update(client2)

    # Parser rsoc
    rsoc_parsed = split_rsoc(client["rsoc"])
    if rsoc_parsed:
        client.update(rsoc_parsed)

    return 1, client


def select_crtf_cli_by_criteria(
    criteria: str,
    criteria_type: str,
    is_actif: bool = True,
    columns: Optional[List[str]] = None
) -> Tuple[int, Dict]:
    """Sélectionne un client certifié par code ou numéro de carte (JOIN gc_cli + gc_cli2).

    Args:
        criteria: Valeur de recherche (code client ou numéro de carte)
        criteria_type: "by_code" ou "by_cardnumber"
        is_actif: Filtrer par statut actif (True/False/None pour tous)
        columns: Colonnes à récupérer (défaut: 22 colonnes standard)

    Returns:
        Tuple (nbrows, client) où:
        - nbrows: 0 ou 1
        - client: Dictionnaire avec les données (vide si non trouvé)

    Raises:
        ValueError: Si criteria_type invalide ou plusieurs clients trouvés
    """
    if columns is None:
        columns = [
            "c.code", "rsoc", "ad1", "ad2", "cpv", "codpost",
            "tel", "portable", "cmpt", "crtf", "birth", "email", "activite",
            "ad_app", "ad_bat", "ad_num", "ad_rue", "ad_ldt",
            "sgmnt", "numcrt", "vnd", "magouv"
        ]

    # Construire la clause WHERE selon le type de critère
    if criteria_type == "by_code":
        where_clause = f"c.code='{criteria}'"
    elif criteria_type == "by_cardnumber":
        where_clause = f"numcrt='{criteria}'"
    else:
        raise ValueError("criteria_type doit être 'by_code' ou 'by_cardnumber'")

    # Requête avec JOIN
    sql = (
        f"SELECT {','.join(columns)} "
        f"FROM gc_cli c, gc_cli2 c2 "
        f"WHERE {where_clause} AND c.code=c2.code AND crtf='1' AND if_oubli='0'"
    )

    if is_actif is not None:
        activite = '1' if is_actif else '0'
        sql += f" AND activite='{activite}'"

    try:
        client = fetch_one(sql)
    except ValueError as e:
        raise ValueError(f"Plusieurs clients trouvés avec le critère '{criteria}' [{criteria_type}]: {e}")

    if client is None:
        return 0, {}

    # Parser rsoc si présent
    if "rsoc" in client:
        rsoc_parsed = split_rsoc(client["rsoc"])
        if rsoc_parsed:
            client.update(rsoc_parsed)

    return 1, client


def select_crtf_cli_by_code(code: str, is_actif: bool = True, columns: Optional[List[str]] = None) -> Tuple[int, Dict]:
    """Sélectionne un client certifié par code (version optimisée avec JOIN).

    Args:
        code: Code du client
        is_actif: Filtrer par statut actif
        columns: Colonnes à récupérer

    Returns:
        Tuple (nbrows, client)
    """
    return select_crtf_cli_by_criteria(code, criteria_type="by_code", is_actif=is_actif, columns=columns)


def select_crtf_cli_by_cardnumber(cardnumber: str, is_actif: bool = True, columns: Optional[List[str]] = None) -> Tuple[int, Dict]:
    """Sélectionne un client certifié par numéro de carte fidélité.

    Args:
        cardnumber: Numéro de carte fidélité
        is_actif: Filtrer par statut actif
        columns: Colonnes à récupérer

    Returns:
        Tuple (nbrows, client)
    """
    return select_crtf_cli_by_criteria(cardnumber, criteria_type="by_cardnumber", is_actif=is_actif, columns=columns)


def get_code_of_crtf_clients() -> Tuple[int, List[Dict]]:
    """Récupère tous les codes des clients certifiés actifs.

    Returns:
        Tuple (nbrows, clients) où clients est une liste de {"code": str}
    """
    sql = (
        "SELECT code FROM gc_cli "
        "WHERE code MATCHES '1*[A-Z]*' AND crtf='1' AND activite='1' AND code<>''"
    )

    clients = fetch_all(sql)
    return len(clients), clients


def select_crtf_from_gccli(is_actif: bool = True, columns: Optional[List[str]] = None) -> Tuple[int, List[Dict]]:
    """Sélectionne tous les clients certifiés de gc_cli (sans gc_cli2).

    Args:
        is_actif: Filtrer par statut actif (True/False/None pour tous)
        columns: Colonnes à récupérer (défaut: colonnes standard)

    Returns:
        Tuple (nbrows, clients) liste de clients
    """
    if columns is None:
        columns = [
            "code", "rsoc", "ad1", "ad2", "cpv", "codpost",
            "tel", "cmpt", "crtf", "birth", "email", "activite"
        ]

    sql = f"SELECT {','.join(columns)} FROM gc_cli WHERE crtf='1'"

    if is_actif is not None:
        activite = '1' if is_actif else '0'
        sql += f" AND activite='{activite}'"

    clients = fetch_all(sql)
    return len(clients), clients


def get_ville(codepost: str) -> Tuple[int, List[Dict]]:
    """Récupère le nom de la ville par code postal.

    Args:
        codepost: Code postal

    Returns:
        Tuple (nbrows, villes) où villes est une liste de {"nom": str}
    """
    sql = f"SELECT nom FROM gc_cdp WHERE code='{codepost}'"

    villes = fetch_all(sql)
    return len(villes), villes


def client_is_crtf_and_actif(code_client: str) -> Tuple[bool, Dict]:
    """Vérifie si un client est actif et certifié.

    Args:
        code_client: Code du client à vérifier

    Returns:
        Tuple (is_valid, client) où:
        - is_valid: True si le client est actif et certifié
        - client: Dictionnaire avec les données du client (vide si non valide)
    """
    nbrows, client = select_crtf_cli(code_client, is_actif=True)
    return nbrows != 0, client


def unload_crtf_cli(columns: List[str], output: str, is_actif: Optional[bool] = True) -> int:
    """Exporte les clients certifiés dans un fichier avec délimiteur '|'.

    Args:
        columns: Liste des colonnes à exporter
        output: Chemin du fichier de sortie
        is_actif: Filtrer par statut actif (True/False/None pour tous)

    Returns:
        Nombre de lignes exportées
    """
    sql = (
        f"SELECT {','.join(columns)} FROM gc_cli "
        f"WHERE code MATCHES '1*[A-Z]*' AND crtf=1"
    )

    if is_actif is not None:
        activite = 1 if is_actif else 0
        sql += f" AND activite={activite}"

    sql += " ORDER BY code"

    return unload_to_file(sql, output, delimiter='|')


def unload_valable_rgpd_cli(output: str) -> int:
    """Exporte les clients RGPD-valides (avec achat depuis 2021).

    Args:
        output: Chemin du fichier de sortie

    Returns:
        Nombre de lignes exportées
    """
    columns = [
        "code", "rsoc", "ad1", "ad2", "cpv", "codpost",
        "tel", "cmpt", "crtf", "birth", "email", "activite"
    ]

    sql = (
        f"SELECT {','.join(columns)} FROM gc_cli "
        f"WHERE gc_cli.crtf = 1 AND gc_cli.code IN ("
        f"  SELECT gc_fac.cli FROM gc_fac WHERE gc_fac.dteinit >= '01/01/2021'"
        f")"
    )

    return unload_to_file(sql, output, delimiter='|')
