"""
Tests unitaires pour invoice_parser.py

Teste le parser de fichiers CSV d'invoices Kiabi (OneTouch).
Équivalent Python de ITFWEBKIA_VENTES_decodekia() en C.
"""

import pytest
from datetime import datetime
from pathlib import Path

from gescokit.partners.kiabi.invoice_parser import (
    InvoiceKiabiLine,
    KiabiInvoiceParser
)


class TestKiabiInvoiceParser:
    """Tests pour la classe KiabiInvoiceParser."""

    def test_parser_initialization(self):
        """Test l'initialisation du parser avec les paramètres par défaut."""
        parser = KiabiInvoiceParser()
        assert parser.encoding == "utf-8"
        assert parser.delimiter == ";"

    def test_parser_initialization_custom(self):
        """Test l'initialisation du parser avec des paramètres personnalisés."""
        parser = KiabiInvoiceParser(encoding="cp850", delimiter=",")
        assert parser.encoding == "cp850"
        assert parser.delimiter == ","

    def test_parse_date_valid(self):
        """Test le parsing d'une date valide."""
        parser = KiabiInvoiceParser()
        date_str = "2025-01-15 14:30:25"
        result = parser._parse_date(date_str)

        assert isinstance(result, datetime)
        assert result.year == 2025
        assert result.month == 1
        assert result.day == 15
        assert result.hour == 14
        assert result.minute == 30
        assert result.second == 25

    def test_parse_date_invalid(self):
        """Test le parsing d'une date invalide (doit retourner datetime.now())."""
        parser = KiabiInvoiceParser()
        date_str = "invalid-date"
        result = parser._parse_date(date_str)

        assert isinstance(result, datetime)
        # Vérifie que c'est une date récente (dans la dernière minute)
        now = datetime.now()
        assert abs((now - result).total_seconds()) < 60

    def test_to_float_valid(self):
        """Test la conversion de chaînes valides en float."""
        parser = KiabiInvoiceParser()

        assert parser._to_float("19.50") == 19.50
        assert parser._to_float("19,50") == 19.50  # Virgule remplacée par point
        assert parser._to_float("0") == 0.0
        assert parser._to_float("") == 0.0

    def test_to_float_invalid(self):
        """Test la conversion de chaînes invalides en float."""
        parser = KiabiInvoiceParser()

        assert parser._to_float("invalid") == 0.0
        assert parser._to_float(None) == 0.0

    def test_to_int_valid(self):
        """Test la conversion de chaînes valides en int."""
        parser = KiabiInvoiceParser()

        assert parser._to_int("123") == 123
        assert parser._to_int("0") == 0
        assert parser._to_int("") == 0

    def test_to_int_invalid(self):
        """Test la conversion de chaînes invalides en int."""
        parser = KiabiInvoiceParser()

        assert parser._to_int("invalid") == 0
        assert parser._to_int("12.34") == 0

    def test_parse_line_basic(self, sample_row_dict):
        """Test le parsing d'une ligne simple sans vouchers."""
        parser = KiabiInvoiceParser()
        line = parser.parse_line(sample_row_dict)

        # Vérifier les identifiants
        assert line.delivery_shop_id == "123"
        assert line.invoice_id == 100001
        assert line.order_id == "1001"
        assert line.order_ref == "ORD001"

        # Vérifier la date
        assert line.invoice_date.year == 2025
        assert line.invoice_date.month == 1
        assert line.invoice_date.day == 15
        assert line.invoice_date.hour == 14
        assert line.invoice_date.minute == 30
        assert line.invoice_date.second == 25

        # Vérifier le client
        assert line.customer_lastname == "DUPONT"
        assert line.customer_firstname == "Jean"
        assert line.loyalty_card_id == "1234567890"
        assert line.loyalty_discount_flag == 1

        # Vérifier le produit
        assert line.product_name == "T-Shirt Bleu"
        assert line.product_ref == "TSH001"
        assert line.product_ean13 == "3456789012345"
        assert line.product_unit_price_ttc == 19.00
        assert line.product_qty == 2.0
        assert line.product_total_price_ttc == 38.00

        # Vérifier les totaux
        assert line.invoice_product_ttc == 38.00

        # Vérifier le paiement
        assert line.payment_method == "Paiement CB"
        assert line.payment_amount == 38.00

        # Vérifier la devise
        assert line.currency == "EUR"
        assert line.currency_rate_eur == 1.0

        # Vérifier les vouchers (doivent être vides)
        assert line.voucher_id_1 == ""
        assert line.voucher_amount_1 == 0.0
        assert line.voucher_id_2 == ""
        assert line.voucher_amount_2 == 0.0
        assert line.voucher_id_3 == ""
        assert line.voucher_amount_3 == 0.0

        # Vérifier la réduction calculée
        assert line.reduction_specif_web == 0.0

    def test_parse_line_with_vouchers(self, sample_row_dict_with_vouchers):
        """Test le parsing d'une ligne avec vouchers."""
        parser = KiabiInvoiceParser()
        line = parser.parse_line(sample_row_dict_with_vouchers)

        # Vérifier les identifiants
        assert line.order_ref == "ORD003"
        assert line.customer_lastname == "BERNARD"

        # Vérifier les vouchers
        assert line.voucher_id_1 == "AVOIR002"
        assert line.voucher_amount_1 == 20.00
        assert line.voucher_id_2 == "AVOIR003"
        assert line.voucher_amount_2 == 15.00
        assert line.voucher_id_3 == ""
        assert line.voucher_amount_3 == 0.0

        # Vérifier la réduction totale et calculée
        assert line.invoice_reduction_ttc == 35.00
        # reduction_specif_web = 35.00 - (20.00 + 15.00) = 0.00
        assert line.reduction_specif_web == 0.0

    def test_parse_line_truncates_long_strings(self):
        """Test que le parser tronque les chaînes trop longues."""
        parser = KiabiInvoiceParser()

        row = {
            "Shop_id": "123",
            "Invoice_id": "100001",
            "Invoice_date": "2025-01-15 14:30:25",
            "Order_id": "1001",
            "Order_ref": "VERY_LONG_ORDER_REF_THAT_EXCEEDS_MAX_LENGTH",  # Max 9
            "Customer_lastname": "VERY_LONG_LASTNAME_THAT_EXCEEDS_25_CHARS",  # Max 25
            "Customer_firstname": "VERY_LONG_FIRSTNAME_THAT_EXCEEDS_25_CHARS",  # Max 25
            "Loyalty_card_id": "1234567890123456789",  # Max 12
            "Loyalty_discount_flag": "0",
            "Product_id": "P001",
            "Product_name": "VERY_LONG_PRODUCT_NAME_THAT_DEFINITELY_EXCEEDS_THE_MAX_LENGTH_OF_60_CHARACTERS",  # Max 60
            "Product_ref": "VERY_LONG_REF_EXCEEDS",  # Max 12
            "Product_ean13": "12345678901234567890",  # Max 13
            "Product_unit_price_ht": "15.83",
            "Product_unit_price_ttc": "19.00",
            "Product_tax_rate": "20.00",
            "Product_qty": "2",
            "Product_total_price_ht": "31.66",
            "Product_total_price_ttc": "38.00",
            "Product_stock_shop_id": "123",
            "Invoice_product_ht": "31.66",
            "Invoice_tax": "6.34",
            "Invoice_product_ttc": "38.00",
            "Payment_method": "Paiement CB",
            "Payment_amount": "38.00",
            "Payment_transaction_id": "123-TRX001",
            "Payment_bank_remittance_date": "2025-01-16",
            "Invoice_reduction_ttc": "0.00",
            "Currency": "EUR",
            "CurrencyRateEur": "1.0",
            "Voucher_id_1": "VERY_LONG_VOUCHER_ID_EXCEEDS_15_CHARS",  # Max 15
            "Voucher_amount_1": "20.00",
            "Voucher_id_2": "",
            "Voucher_amount_2": "",
            "Voucher_id_3": "",
            "Voucher_amount_3": "",
        }

        line = parser.parse_line(row)

        # Vérifier les troncatures
        assert len(line.order_ref) <= 9
        assert len(line.customer_lastname) <= 25
        assert len(line.customer_firstname) <= 25
        assert len(line.loyalty_card_id) <= 12
        assert len(line.product_name) <= 60
        assert len(line.product_ref) <= 12
        assert len(line.product_ean13) <= 13
        assert len(line.voucher_id_1) <= 15

    def test_parse_file(self, sample_csv_file):
        """Test le parsing d'un fichier CSV complet."""
        parser = KiabiInvoiceParser()
        lines = parser.parse_file(sample_csv_file)

        # Vérifier le nombre de lignes
        assert len(lines) == 4

        # Vérifier que toutes les lignes sont des InvoiceKiabiLine
        for line in lines:
            assert isinstance(line, InvoiceKiabiLine)

        # Vérifier quelques valeurs de la première ligne
        assert lines[0].order_ref == "ORD001"
        assert lines[0].customer_lastname == "DUPONT"

        # Vérifier la dernière ligne (avec vouchers)
        assert lines[3].order_ref == "ORD003"
        assert lines[3].voucher_id_1 == "AVOIR002"
        assert lines[3].voucher_amount_1 == 20.00

    def test_parse_file_not_found(self, tmp_path):
        """Test le parsing d'un fichier inexistant."""
        parser = KiabiInvoiceParser()
        non_existent_file = tmp_path / "non_existent.csv"

        with pytest.raises(FileNotFoundError):
            parser.parse_file(non_existent_file)

    def test_group_by_invoice(self, sample_csv_file):
        """Test le regroupement des lignes par facture."""
        parser = KiabiInvoiceParser()
        lines = parser.parse_file(sample_csv_file)
        invoices = parser.group_by_invoice(lines)

        # Vérifier le nombre de factures distinctes
        assert len(invoices) == 3

        # Vérifier les clés (order_ref)
        assert "ORD001" in invoices
        assert "ORD002" in invoices
        assert "ORD003" in invoices

        # Vérifier le nombre de lignes par facture
        assert len(invoices["ORD001"]) == 1  # 1 ligne
        assert len(invoices["ORD002"]) == 2  # 2 lignes
        assert len(invoices["ORD003"]) == 1  # 1 ligne

    def test_currency_default_value(self):
        """Test que la devise par défaut est EUR."""
        parser = KiabiInvoiceParser()

        row = {
            "Shop_id": "123",
            "Invoice_id": "100001",
            "Invoice_date": "2025-01-15 14:30:25",
            "Order_id": "1001",
            "Order_ref": "ORD001",
            "Customer_lastname": "DUPONT",
            "Customer_firstname": "Jean",
            "Loyalty_card_id": "1234567890",
            "Loyalty_discount_flag": "1",
            "Product_id": "P001",
            "Product_name": "T-Shirt Bleu",
            "Product_ref": "TSH001",
            "Product_ean13": "3456789012345",
            "Product_unit_price_ht": "15.83",
            "Product_unit_price_ttc": "19.00",
            "Product_tax_rate": "20.00",
            "Product_qty": "2",
            "Product_total_price_ht": "31.66",
            "Product_total_price_ttc": "38.00",
            "Product_stock_shop_id": "123",
            "Invoice_product_ht": "31.66",
            "Invoice_tax": "6.34",
            "Invoice_product_ttc": "38.00",
            "Payment_method": "Paiement CB",
            "Payment_amount": "38.00",
            "Payment_transaction_id": "123-TRX001",
            "Payment_bank_remittance_date": "2025-01-16",
            "Invoice_reduction_ttc": "0.00",
            "Currency": "",  # Vide
            "CurrencyRateEur": "",  # Vide
            "Voucher_id_1": "",
            "Voucher_amount_1": "",
            "Voucher_id_2": "",
            "Voucher_amount_2": "",
            "Voucher_id_3": "",
            "Voucher_amount_3": "",
        }

        line = parser.parse_line(row)

        # Vérifier les valeurs par défaut
        assert line.currency == "EUR"
        assert line.currency_rate_eur == 1.0

    def test_reduction_specif_web_calculation(self):
        """Test le calcul de la réduction spécifique web."""
        parser = KiabiInvoiceParser()

        # Cas 1: Réduction totale = avoirs → reduction_specif_web = 0
        row1 = {
            "Shop_id": "123", "Invoice_id": "100001", "Invoice_date": "2025-01-15 14:30:25",
            "Order_id": "1001", "Order_ref": "ORD001", "Customer_lastname": "DUPONT",
            "Customer_firstname": "Jean", "Loyalty_card_id": "", "Loyalty_discount_flag": "0",
            "Product_id": "P001", "Product_name": "Test", "Product_ref": "TST001",
            "Product_ean13": "1234567890123", "Product_unit_price_ht": "10.00",
            "Product_unit_price_ttc": "12.00", "Product_tax_rate": "20.00",
            "Product_qty": "1", "Product_total_price_ht": "10.00",
            "Product_total_price_ttc": "12.00", "Product_stock_shop_id": "123",
            "Invoice_product_ht": "10.00", "Invoice_tax": "2.00", "Invoice_product_ttc": "12.00",
            "Payment_method": "CB", "Payment_amount": "12.00", "Payment_transaction_id": "TRX001",
            "Payment_bank_remittance_date": "2025-01-16", "Invoice_reduction_ttc": "30.00",
            "Currency": "EUR", "CurrencyRateEur": "1.0",
            "Voucher_id_1": "AV001", "Voucher_amount_1": "20.00",
            "Voucher_id_2": "AV002", "Voucher_amount_2": "10.00",
            "Voucher_id_3": "", "Voucher_amount_3": "",
        }
        line1 = parser.parse_line(row1)
        assert line1.reduction_specif_web == 0.0  # 30 - (20 + 10) = 0

        # Cas 2: Réduction totale > avoirs → reduction_specif_web > 0
        row2 = row1.copy()
        row2["Invoice_reduction_ttc"] = "50.00"
        line2 = parser.parse_line(row2)
        assert line2.reduction_specif_web == 20.0  # 50 - (20 + 10) = 20
