"""
Parser pour les fichiers CSV de slips (avoirs) Kiabi (OneTouch).

Ce module permet de parser les fichiers CSV de slips (avoirs) générés par le
système OneTouch et utilisés dans le système GCV (via itfwebkia.ec).

Format de fichier: slip-XXX-YYYYMMDD-AAAAAAAAA.csv
"""

import csv
import structlog
from dataclasses import dataclass, field
from datetime import datetime
from pathlib import Path
from typing import Optional, List, Dict, Any

logger = structlog.get_logger()


@dataclass
class SlipKiabiLine:
    """
    Représente une ligne de slip (avoir) Kiabi (équivalent de ttslip en C).

    Correspond aux champs extraits dans ITFWEBKIA_AVOIRS_decodekia() du fichier
    itfwebkia.ec lignes 3358-3431.
    """
    # Identifiants
    shop_id: str                    # Shop_id
    slip_id: str                    # Slip_id (nocom) - max 15 caractères
    slip_date: datetime             # Slip_date (dtcom + hrcom)

    # Facture associée
    invoice_id: int                 # Invoice_id (invoicekia_id)
    invoice_date: datetime          # Invoice_date (invoicekia_date)
    order_id: str                   # Order_id
    order_ref: str                  # Order_ref (refcom)

    # Client
    customer_lastname: str          # Customer_lastname (nomcli)
    customer_firstname: str         # Customer_firstname (prncli)
    loyalty_card_id: str            # Loyalty_card_id (numcrt)
    loyalty_discount_flag: int      # Loyalty_discount_flag (ilyaremise)

    # Produit
    product_id: str                 # Product_id
    product_name: str               # Product_name (desig)
    product_ref: str                # Product_ref (code)
    product_ean13: str              # Product_ean13 (ean)
    product_unit_price_ht: float    # Product_unit_price_ht (pvht)
    product_unit_price_ttc: float   # Product_unit_price_ttc
    product_tax_rate: float         # Product_tax_rate (txtva)
    product_qty: float              # Product_qty (qt) - quantité retournée
    product_total_price_ht: float   # Product_total_price_ht
    product_total_price_ttc: float  # Product_total_price_ttc

    # Totaux slip (se répètent pour chaque ligne)
    slip_product_ht: float          # Slip_product_ht
    slip_tax: float                 # Slip_tax
    slip_product_ttc: float         # Slip_product_ttc

    # Remboursement
    refund_method: str              # Refund_method

    # Champs avec valeurs par défaut (doivent être à la fin)
    # Référence interne (calculée: order_ref + "-" + slip_id)
    slip_ref: str = ""              # refercom (usage interne)

    # Données brutes pour référence
    raw_data: Dict[str, str] = field(default_factory=dict)


class KiabiSlipParser:
    """
    Parser pour les fichiers CSV de slips (avoirs) Kiabi.

    Équivalent Python de la fonction ITFWEBKIA_AVOIRS_decodekia() du fichier C
    (itfwebkia.ec lignes 3358-3431).
    """

    # Colonnes attendues dans le CSV (ordre exact du fichier - voir ligne 3300-3328)
    EXPECTED_COLUMNS = [
        "Shop_id",
        "Slip_id",
        "Slip_date",
        "Invoice_id",
        "Invoice_date",
        "Order_id",
        "Order_ref",
        "Customer_lastname",
        "Customer_firstname",
        "Loyalty_card_id",
        "Loyalty_discount_flag",
        "Product_id",
        "Product_name",
        "Product_ref",
        "Product_ean13",
        "Product_unit_price_ht",
        "Product_unit_price_ttc",
        "Product_tax_rate",
        "Product_qty",
        "Product_total_price_ht",
        "Product_total_price_ttc",
        "Slip_product_ht",
        "Slip_tax",
        "Slip_product_ttc",
        "Refund_method"
    ]

    def __init__(self, encoding: str = "utf-8", delimiter: str = ";"):
        """
        Initialise le parser.

        Parameters
        ----------
        encoding : str, default="utf-8"
            Encodage du fichier CSV (utf-8 par défaut, cp850 en C)
        delimiter : str, default=";"
            Séparateur de colonnes (point-virgule par défaut)
        """
        self.encoding = encoding
        self.delimiter = delimiter

    @staticmethod
    def _parse_date(date_str: str) -> datetime:
        """
        Parse une date au format ISO: 2020-04-24 00:59:58.

        Correspond au parsing dans ITFWEBKIA_AVOIRS_decodekia ligne 3371-3377.

        Parameters
        ----------
        date_str : str
            Date au format "YYYY-MM-DD HH:MM:SS"

        Returns
        -------
        datetime
            Date parsée
        """
        try:
            return datetime.strptime(date_str.strip(), "%Y-%m-%d %H:%M:%S")
        except ValueError as e:
            logger.warning(f"Date invalide: {date_str}, erreur: {e}")
            return datetime.now()

    @staticmethod
    def _parse_date_only(date_str: str) -> datetime:
        """
        Parse une date sans heure au format: 2020-04-24.

        Utilisé pour invoice_date (ligne 3381-3384).

        Parameters
        ----------
        date_str : str
            Date au format "YYYY-MM-DD"

        Returns
        -------
        datetime
            Date parsée (heure à 00:00:00)
        """
        try:
            # Extraire juste la date si format complet fourni
            if " " in date_str:
                date_str = date_str.split()[0]
            return datetime.strptime(date_str.strip(), "%Y-%m-%d")
        except ValueError as e:
            logger.warning(f"Date invalide: {date_str}, erreur: {e}")
            return datetime.now()

    @staticmethod
    def _to_float(value: str) -> float:
        """
        Convertit une chaîne en float, gère les virgules et points.

        Correspond à strtodot() en C.

        Parameters
        ----------
        value : str
            Valeur à convertir

        Returns
        -------
        float
            Valeur convertie
        """
        if not value or value.strip() == "":
            return 0.0
        try:
            return float(value.strip().replace(",", "."))
        except ValueError:
            logger.warning(f"Impossible de convertir en float: {value}")
            return 0.0

    @staticmethod
    def _to_int(value: str) -> int:
        """
        Convertit une chaîne en int.

        Parameters
        ----------
        value : str
            Valeur à convertir

        Returns
        -------
        int
            Valeur convertie
        """
        if not value or value.strip() == "":
            return 0
        try:
            return int(value.strip())
        except ValueError:
            logger.warning(f"Impossible de convertir en int: {value}")
            return 0

    def parse_line(self, row: Dict[str, str]) -> SlipKiabiLine:
        """
        Parse une ligne du CSV et retourne un objet SlipKiabiLine.

        Équivalent de ITFWEBKIA_AVOIRS_decodekia() (lignes 3358-3431).

        Parameters
        ----------
        row : Dict[str, str]
            Dictionnaire représentant une ligne du CSV

        Returns
        -------
        SlipKiabiLine
            Objet représentant la ligne parsée
        """
        # Parse des champs de base
        slip_id = row.get("Slip_id", "").strip()[:15]
        order_ref = row.get("Order_ref", "").strip()[:9]

        # Calcul de slip_ref (refercom = order_ref + "-" + slip_id)
        # Correspond aux lignes 3390-3393 de itfwebkia.ec
        slip_ref = f"{order_ref}-{slip_id}".strip()

        return SlipKiabiLine(
            shop_id=row.get("Shop_id", "").strip(),
            slip_id=slip_id,
            slip_date=self._parse_date(row.get("Slip_date", "")),

            invoice_id=self._to_int(row.get("Invoice_id", "0")),
            invoice_date=self._parse_date_only(row.get("Invoice_date", "")),
            order_id=row.get("Order_id", "").strip(),
            order_ref=order_ref,

            slip_ref=slip_ref,

            customer_lastname=row.get("Customer_lastname", "").strip()[:25],
            customer_firstname=row.get("Customer_firstname", "").strip()[:25],
            loyalty_card_id=row.get("Loyalty_card_id", "").strip()[:12],
            loyalty_discount_flag=self._to_int(row.get("Loyalty_discount_flag", "0")),

            product_id=row.get("Product_id", "").strip(),
            product_name=row.get("Product_name", "").strip()[:60],
            product_ref=row.get("Product_ref", "").strip()[:12],
            product_ean13=row.get("Product_ean13", "").strip()[:13],
            product_unit_price_ht=self._to_float(row.get("Product_unit_price_ht", "0")),
            product_unit_price_ttc=self._to_float(row.get("Product_unit_price_ttc", "0")),
            product_tax_rate=self._to_float(row.get("Product_tax_rate", "0")),
            product_qty=self._to_float(row.get("Product_qty", "0")),
            product_total_price_ht=self._to_float(row.get("Product_total_price_ht", "0")),
            product_total_price_ttc=self._to_float(row.get("Product_total_price_ttc", "0")),

            slip_product_ht=self._to_float(row.get("Slip_product_ht", "0")),
            slip_tax=self._to_float(row.get("Slip_tax", "0")),
            slip_product_ttc=self._to_float(row.get("Slip_product_ttc", "0")),

            refund_method=row.get("Refund_method", "").strip()[:80],

            raw_data=row
        )

    def parse_file(self, file_path: Path) -> List[SlipKiabiLine]:
        """
        Parse un fichier CSV complet et retourne la liste des lignes.

        Correspond à la lecture du fichier dans ITFWEBKIA_AVOIRS_getrefav().

        Parameters
        ----------
        file_path : Path
            Chemin vers le fichier CSV

        Returns
        -------
        List[SlipKiabiLine]
            Liste des lignes parsées
        """
        logger.info(f"Parsing du fichier slip: {file_path}")

        if not file_path.exists():
            logger.error(f"Fichier introuvable: {file_path}")
            raise FileNotFoundError(f"Fichier introuvable: {file_path}")

        lines = []

        with open(file_path, "r", encoding=self.encoding) as csvfile:
            reader = csv.DictReader(csvfile, delimiter=self.delimiter)

            # Vérifier que les colonnes attendues sont présentes
            if reader.fieldnames:
                missing_cols = set(self.EXPECTED_COLUMNS) - set(reader.fieldnames)
                if missing_cols:
                    logger.warning(f"Colonnes manquantes: {missing_cols}")

            for row_num, row in enumerate(reader, start=2):  # start=2 car ligne 1 = header
                try:
                    line = self.parse_line(row)
                    lines.append(line)
                except Exception as e:
                    logger.error(f"Erreur ligne {row_num}: {e}", exc_info=True)

        logger.info(f"Fichier slip parsé: {len(lines)} lignes trouvées")
        return lines

    def group_by_slip(self, lines: List[SlipKiabiLine]) -> Dict[str, List[SlipKiabiLine]]:
        """
        Regroupe les lignes par numéro de slip (slip_ref).

        Correspond à la logique dans ITFWEBKIA_AVOIRS qui traite chaque
        slip individuellement.

        Parameters
        ----------
        lines : List[SlipKiabiLine]
            Liste des lignes parsées

        Returns
        -------
        Dict[str, List[SlipKiabiLine]]
            Dictionnaire {slip_ref: [lignes]}
        """
        slips = {}
        for line in lines:
            if line.slip_ref not in slips:
                slips[line.slip_ref] = []
            slips[line.slip_ref].append(line)

        logger.info(f"Nombre de slips distincts: {len(slips)}")
        return slips
