"""
Interface CLI pour les slips (avoirs) Kiabi.

Commandes disponibles pour les fichiers slip:
- parse-slip: Parser un fichier CSV de slips
- slip-info: Afficher des informations sur un fichier CSV de slips
- slip-doc: Afficher la documentation du format slip
"""

import typer
import structlog
from pathlib import Path
from rich.console import Console
from rich.table import Table

from .slip_parser import KiabiSlipParser

logger = structlog.get_logger()
console = Console()

slip_app = typer.Typer(
    help="Commandes pour gérer les fichiers de slips (avoirs) Kiabi",
    no_args_is_help=True
)


@slip_app.command("parse")
def parse_slip(
    csv_file: Path = typer.Argument(
        ...,
        help="Chemin vers le fichier CSV de slips",
        exists=True,
        file_okay=True,
        dir_okay=False,
        resolve_path=True
    ),
    show_lines: bool = typer.Option(
        False,
        "--show-lines",
        help="Afficher toutes les lignes parsées"
    ),
    limit: int = typer.Option(
        10,
        "--limit",
        "-n",
        help="Nombre maximum de lignes à afficher"
    )
):
    """
    Parse un fichier CSV de slips (avoirs) Kiabi et affiche le résultat.

    Examples
    --------
    gescokit kiabi slip parse slip-123-20250101-XXXXX.csv
    gescokit kiabi slip parse slip-123-20250101-XXXXX.csv --show-lines --limit 20
    """
    try:
        parser = KiabiSlipParser()
        lines = parser.parse_file(csv_file)

        console.print(f"\n[bold green]✓[/bold green] Fichier slip parsé: {csv_file}")
        console.print(f"  Nombre de lignes: {len(lines)}")

        # Grouper par slip
        slips = parser.group_by_slip(lines)
        console.print(f"  Nombre de slips (avoirs): {len(slips)}")

        if show_lines:
            console.print(f"\n[bold]Premières lignes (max {limit}):[/bold]")
            for i, line in enumerate(lines[:limit], start=1):
                console.print(f"\n[bold]Ligne {i}:[/bold]")
                console.print(f"  Slip ref: {line.slip_ref}")
                console.print(f"  Slip date: {line.slip_date}")
                console.print(f"  Invoice ref: {line.order_ref} (ID: {line.invoice_id})")
                console.print(f"  Customer: {line.customer_firstname} {line.customer_lastname}")
                console.print(f"  Product: {line.product_name} ({line.product_ref})")
                console.print(f"  Quantity returned: {line.product_qty}")
                console.print(f"  Total TTC: {line.product_total_price_ttc:.2f}")
                console.print(f"  Refund method: {line.refund_method}")

    except Exception as e:
        console.print(f"\n[bold red]✗[/bold red] Erreur: {e}")
        raise typer.Exit(code=1)


@slip_app.command("info")
def show_slip_info(
    csv_file: Path = typer.Argument(
        ...,
        help="Chemin vers le fichier CSV de slips",
        exists=True,
        file_okay=True,
        dir_okay=False,
        resolve_path=True
    )
):
    """
    Affiche des informations détaillées sur un fichier CSV de slips.

    Examples
    --------
    gescokit kiabi slip info slip-123-20250101-XXXXX.csv
    """
    try:
        parser = KiabiSlipParser()
        lines = parser.parse_file(csv_file)
        slips = parser.group_by_slip(lines)

        # Statistiques générales
        console.print(f"\n[bold]Informations sur le fichier slip:[/bold] {csv_file}")
        console.print(f"  Nombre de lignes: {len(lines)}")
        console.print(f"  Nombre de slips (avoirs): {len(slips)}")

        # Statistiques par slip
        table = Table(title="\nRésumé des slips")
        table.add_column("Slip Ref", style="cyan")
        table.add_column("Date", style="green")
        table.add_column("Client", style="yellow")
        table.add_column("Nb lignes", justify="right")
        table.add_column("Total TTC", justify="right", style="bold")
        table.add_column("Refund", style="magenta")

        for slip_ref, slip_lines in list(slips.items())[:20]:  # Limiter à 20
            first_line = slip_lines[0]
            total_ttc = first_line.slip_product_ttc  # Total du slip (identique sur toutes les lignes)

            table.add_row(
                slip_ref,
                first_line.slip_date.strftime("%Y-%m-%d %H:%M"),
                f"{first_line.customer_firstname} {first_line.customer_lastname}",
                str(len(slip_lines)),
                f"{total_ttc:.2f}",
                first_line.refund_method[:20]  # Tronquer pour l'affichage
            )

        console.print(table)

        if len(slips) > 20:
            console.print(f"\n[dim]... et {len(slips) - 20} autres slips[/dim]")

        # Statistiques sur les méthodes de remboursement
        refund_methods = {}
        for line in lines:
            method = line.refund_method
            if method not in refund_methods:
                refund_methods[method] = 0
            refund_methods[method] += 1

        if refund_methods:
            console.print(f"\n[bold]Méthodes de remboursement:[/bold]")
            for method, count in refund_methods.items():
                console.print(f"  {method}: {count} lignes")

    except Exception as e:
        console.print(f"\n[bold red]✗[/bold red] Erreur: {e}")
        raise typer.Exit(code=1)


@slip_app.command("doc")
def show_slip_documentation():
    """
    Affiche la documentation du format CSV de slips Kiabi.
    """
    doc_text = """
[bold]Format CSV des slips (avoirs) Kiabi (OneTouch)[/bold]

[bold cyan]Nomenclature des fichiers:[/bold cyan]
  slip-XXX-YYYYMMDD-AAAAAAAAA.csv

  Où:
    XXX        : Identifiant du magasin
    YYYYMMDD   : Date du slip
    AAAAAAAAA  : Identifiant unique

[bold cyan]Structure du fichier:[/bold cyan]
  Format   : CSV avec séparateur point-virgule (;)
  Encodage : UTF-8
  Header   : Première ligne contient les noms de colonnes

[bold cyan]Colonnes principales:[/bold cyan]

Entête:
  • Shop_id                  : Identifiant OneTouch de magasin
  • Slip_id                  : Identifiant d'avoir (unique, max 15 car.)
  • Slip_date                : Date d'émission avoir (YYYY-MM-DD HH:MM:SS)
  • Invoice_id               : Identifiant facture associée
  • Invoice_date             : Date facture associée
  • Order_id                 : Identifiant commande associée
  • Order_ref                : Référence commande associée (max 9 car.)
  • Customer_lastname        : Nom du client
  • Customer_firstname       : Prénom du client
  • Loyalty_card_id          : Numéro carte fidélité (max 12 chiffres)
  • Loyalty_discount_flag    : Utilisation remise 15% (0=non, 1=oui)

Produits:
  • Product_id               : Identifiant produit
  • Product_name             : Nom du produit (max 60 car.)
  • Product_ref              : Référence produit (max 12 car.)
  • Product_ean13            : Code barre EAN13 (max 13 car.)
  • Product_unit_price_ht    : Prix unitaire HT
  • Product_unit_price_ttc   : Prix unitaire TTC
  • Product_tax_rate         : Taux TVA
  • Product_qty              : Quantité retournée
  • Product_total_price_ht   : Montant HT remboursé (produit × quantité)
  • Product_total_price_ttc  : Montant TTC remboursé (produit × quantité)

Pied (se répète pour chaque ligne):
  • Slip_product_ht          : Montant total HT de l'avoir
  • Slip_tax                 : Montant TVA total de l'avoir
  • Slip_product_ttc         : Montant total TTC de l'avoir
  • Refund_method            : Mode remboursement (bon réduction ou CB)

[bold cyan]Champ calculé:[/bold cyan]
  • slip_ref                 : Référence interne = Order_ref + "-" + Slip_id

[bold cyan]Correspondance avec le code C:[/bold cyan]
  Ce parser Python est équivalent à la fonction C :
    ITFWEBKIA_AVOIRS_decodekia()
  dans le fichier itfwebkia.ec (lignes 3358-3431)

[bold cyan]Références:[/bold cyan]
  • Fonction C  : ITFWEBKIA_AVOIRS_decodekia()
  • Structure C : ttslip
  • Fichier     : common_src/itfwebkia.ec (lignes 3295-3431)
"""
    console.print(doc_text)


if __name__ == "__main__":
    slip_app()
