"""
Interface CLI pour le plugin Kiabi de gescokit.

Commandes disponibles:
- parse: Parser un fichier CSV d'invoices
- to-excel: Convertir un fichier CSV vers Excel
- to-sqlite: Convertir un fichier CSV vers SQLite
- to-sqlite-bulk: Convertir plusieurs fichiers CSV vers une base SQLite
- info: Afficher des informations sur un fichier CSV
- slip: Sous-menu pour les slips (avoirs)
"""

import typer
import structlog
from pathlib import Path
from typing import Optional, List
from rich.console import Console
from rich.table import Table
from rich import print as rprint

from .invoice_parser import KiabiInvoiceParser
from .converter import KiabiInvoiceConverter
from .cli_slip import slip_app

logger = structlog.get_logger()
console = Console()

app = typer.Typer(
    help="Commandes pour gérer les fichiers d'invoices et slips Kiabi (OneTouch)",
    no_args_is_help=True
)

# Ajouter le sous-menu slip
app.add_typer(slip_app, name="slip", help="Commandes pour les slips (avoirs)")


@app.command("parse")
def parse_invoice(
    csv_file: Path = typer.Argument(
        ...,
        help="Chemin vers le fichier CSV d'invoices",
        exists=True,
        file_okay=True,
        dir_okay=False,
        resolve_path=True
    ),
    show_lines: bool = typer.Option(
        False,
        "--show-lines",
        help="Afficher toutes les lignes parsées"
    ),
    limit: int = typer.Option(
        10,
        "--limit",
        "-n",
        help="Nombre maximum de lignes à afficher"
    )
):
    """
    Parse un fichier CSV d'invoices Kiabi et affiche le résultat.

    Examples
    --------
    gescokit kiabi parse invoice-123-20250101-XXXXX.csv
    gescokit kiabi parse invoice-123-20250101-XXXXX.csv --show-lines --limit 20
    """
    try:
        parser = KiabiInvoiceParser()
        lines = parser.parse_file(csv_file)

        console.print(f"\n[bold green]✓[/bold green] Fichier parsé: {csv_file}")
        console.print(f"  Nombre de lignes: {len(lines)}")

        # Grouper par facture
        invoices = parser.group_by_invoice(lines)
        console.print(f"  Nombre de factures: {len(invoices)}")

        if show_lines:
            console.print(f"\n[bold]Premières lignes (max {limit}):[/bold]")
            for i, line in enumerate(lines[:limit], start=1):
                console.print(f"\n[bold]Ligne {i}:[/bold]")
                console.print(f"  Order ref: {line.order_ref}")
                console.print(f"  Invoice date: {line.invoice_date}")
                console.print(f"  Customer: {line.customer_firstname} {line.customer_lastname}")
                console.print(f"  Product: {line.product_name} ({line.product_ref})")
                console.print(f"  Quantity: {line.product_qty}")
                console.print(f"  Total TTC: {line.product_total_price_ttc:.2f} {line.currency}")

                # Afficher les vouchers s'ils sont présents
                vouchers = []
                if line.voucher_id_1:
                    vouchers.append(f"{line.voucher_id_1}: {line.voucher_amount_1:.2f}")
                if line.voucher_id_2:
                    vouchers.append(f"{line.voucher_id_2}: {line.voucher_amount_2:.2f}")
                if line.voucher_id_3:
                    vouchers.append(f"{line.voucher_id_3}: {line.voucher_amount_3:.2f}")

                if vouchers:
                    console.print(f"  Vouchers: {', '.join(vouchers)}")

    except Exception as e:
        console.print(f"\n[bold red]✗[/bold red] Erreur: {e}")
        raise typer.Exit(code=1)


@app.command("to-excel")
def convert_to_excel(
    csv_file: Path = typer.Argument(
        ...,
        help="Chemin vers le fichier CSV d'invoices",
        exists=True,
        file_okay=True,
        dir_okay=False,
        resolve_path=True
    ),
    output: Optional[Path] = typer.Option(
        None,
        "--output",
        "-o",
        help="Chemin vers le fichier Excel de sortie (auto si non fourni)"
    ),
    sheet_name: str = typer.Option(
        "Invoices",
        "--sheet-name",
        "-s",
        help="Nom de la feuille Excel"
    )
):
    """
    Convertit un fichier CSV d'invoices Kiabi vers Excel.

    Examples
    --------
    gescokit kiabi to-excel invoice-123-20250101-XXXXX.csv
    gescokit kiabi to-excel invoice-123-20250101-XXXXX.csv -o result.xlsx
    """
    try:
        converter = KiabiInvoiceConverter()
        output_file = converter.to_excel(csv_file, output, sheet_name)

        console.print(f"\n[bold green]✓[/bold green] Fichier Excel créé: {output_file}")

    except ImportError as e:
        console.print(f"\n[bold red]✗[/bold red] {e}")
        raise typer.Exit(code=1)
    except Exception as e:
        console.print(f"\n[bold red]✗[/bold red] Erreur: {e}")
        raise typer.Exit(code=1)


@app.command("to-sqlite")
def convert_to_sqlite(
    csv_file: Path = typer.Argument(
        ...,
        help="Chemin vers le fichier CSV d'invoices",
        exists=True,
        file_okay=True,
        dir_okay=False,
        resolve_path=True
    ),
    output: Optional[Path] = typer.Option(
        None,
        "--output",
        "-o",
        help="Chemin vers la base SQLite de sortie (auto si non fourni)"
    ),
    overwrite: bool = typer.Option(
        False,
        "--overwrite",
        help="Écraser la table existante"
    )
):
    """
    Convertit un fichier CSV d'invoices Kiabi vers SQLite.

    Examples
    --------
    gescokit kiabi to-sqlite invoice-123-20250101-XXXXX.csv
    gescokit kiabi to-sqlite invoice-123-20250101-XXXXX.csv -o invoices.db --overwrite
    """
    try:
        converter = KiabiInvoiceConverter()
        db_file = converter.to_sqlite(csv_file, output, overwrite)

        console.print(f"\n[bold green]✓[/bold green] Base SQLite créée: {db_file}")

    except ImportError as e:
        console.print(f"\n[bold red]✗[/bold red] {e}")
        raise typer.Exit(code=1)
    except Exception as e:
        console.print(f"\n[bold red]✗[/bold red] Erreur: {e}")
        raise typer.Exit(code=1)


@app.command("to-sqlite-bulk")
def convert_to_sqlite_bulk(
    csv_files: List[Path] = typer.Argument(
        ...,
        help="Chemins vers les fichiers CSV d'invoices"
    ),
    output: Path = typer.Option(
        ...,
        "--output",
        "-o",
        help="Chemin vers la base SQLite de sortie"
    ),
    overwrite: bool = typer.Option(
        False,
        "--overwrite",
        help="Écraser la table existante"
    )
):
    """
    Convertit plusieurs fichiers CSV d'invoices Kiabi vers une base SQLite.

    Examples
    --------
    gescokit kiabi to-sqlite-bulk invoice-*.csv -o invoices.db
    gescokit kiabi to-sqlite-bulk invoice-1.csv invoice-2.csv -o db.sqlite --overwrite
    """
    try:
        # Vérifier que tous les fichiers existent
        for csv_file in csv_files:
            if not csv_file.exists():
                console.print(f"\n[bold red]✗[/bold red] Fichier introuvable: {csv_file}")
                raise typer.Exit(code=1)

        converter = KiabiInvoiceConverter()
        db_file = converter.to_sqlite_bulk(csv_files, output, overwrite)

        console.print(f"\n[bold green]✓[/bold green] Base SQLite créée: {db_file}")
        console.print(f"  Nombre de fichiers traités: {len(csv_files)}")

    except ImportError as e:
        console.print(f"\n[bold red]✗[/bold red] {e}")
        raise typer.Exit(code=1)
    except Exception as e:
        console.print(f"\n[bold red]✗[/bold red] Erreur: {e}")
        raise typer.Exit(code=1)


@app.command("info")
def show_info(
    csv_file: Path = typer.Argument(
        ...,
        help="Chemin vers le fichier CSV d'invoices",
        exists=True,
        file_okay=True,
        dir_okay=False,
        resolve_path=True
    )
):
    """
    Affiche des informations détaillées sur un fichier CSV d'invoices.

    Examples
    --------
    gescokit kiabi info invoice-123-20250101-XXXXX.csv
    """
    try:
        parser = KiabiInvoiceParser()
        lines = parser.parse_file(csv_file)
        invoices = parser.group_by_invoice(lines)

        # Statistiques générales
        console.print(f"\n[bold]Informations sur le fichier:[/bold] {csv_file}")
        console.print(f"  Nombre de lignes: {len(lines)}")
        console.print(f"  Nombre de factures: {len(invoices)}")

        # Statistiques par facture
        table = Table(title="\nRésumé des factures")
        table.add_column("Order Ref", style="cyan")
        table.add_column("Date", style="green")
        table.add_column("Client", style="yellow")
        table.add_column("Nb lignes", justify="right")
        table.add_column("Total TTC", justify="right", style="bold")

        for order_ref, invoice_lines in list(invoices.items())[:20]:  # Limiter à 20
            first_line = invoice_lines[0]
            total_ttc = sum(line.product_total_price_ttc for line in invoice_lines)

            table.add_row(
                order_ref,
                first_line.invoice_date.strftime("%Y-%m-%d %H:%M"),
                f"{first_line.customer_firstname} {first_line.customer_lastname}",
                str(len(invoice_lines)),
                f"{total_ttc:.2f} {first_line.currency}"
            )

        console.print(table)

        if len(invoices) > 20:
            console.print(f"\n[dim]... et {len(invoices) - 20} autres factures[/dim]")

        # Statistiques sur les vouchers
        voucher_count = sum(
            1 for line in lines
            if line.voucher_id_1 or line.voucher_id_2 or line.voucher_id_3
        )
        if voucher_count > 0:
            console.print(f"\n[bold]Vouchers:[/bold]")
            console.print(f"  Lignes avec vouchers: {voucher_count}/{len(lines)}")

    except Exception as e:
        console.print(f"\n[bold red]✗[/bold red] Erreur: {e}")
        raise typer.Exit(code=1)


@app.command("doc")
def show_documentation():
    """
    Affiche la documentation du format CSV d'invoices Kiabi.
    """
    doc_text = """
[bold]Format CSV des invoices Kiabi (OneTouch)[/bold]

[bold cyan]Nomenclature des fichiers:[/bold cyan]
  invoice-XXX-YYYYMMDD-AAAAAAAAA.csv

  Où:
    XXX        : Identifiant du magasin
    YYYYMMDD   : Date de la facture
    AAAAAAAAA  : Identifiant unique

[bold cyan]Structure du fichier:[/bold cyan]
  Format   : CSV avec séparateur point-virgule (;)
  Encodage : UTF-8
  Header   : Première ligne contient les noms de colonnes

[bold cyan]Colonnes principales:[/bold cyan]
  • Shop_id                  : Identifiant du magasin de livraison
  • Invoice_id               : Identifiant de la facture
  • Invoice_date             : Date/heure de la facture (YYYY-MM-DD HH:MM:SS)
  • Order_ref                : Référence de commande (max 9 caractères)
  • Customer_lastname        : Nom du client
  • Customer_firstname       : Prénom du client
  • Loyalty_card_id          : Numéro de carte de fidélité (max 12 chiffres)
  • Product_ref              : Référence produit (max 12 caractères)
  • Product_ean13            : Code barre EAN13
  • Product_unit_price_ttc   : Prix unitaire TTC
  • Product_qty              : Quantité
  • Product_total_price_ttc  : Prix total TTC
  • Payment_method           : Mode de paiement
  • Payment_amount           : Montant du paiement
  • Currency                 : Devise (EUR, USD, etc.)

[bold cyan]Colonnes vouchers (ajoutées 21/07/2025):[/bold cyan]
  • Voucher_id_1             : Identifiant avoir #1 (max 15 caractères)
  • Voucher_amount_1         : Montant avoir #1
  • Voucher_id_2             : Identifiant avoir #2
  • Voucher_amount_2         : Montant avoir #2
  • Voucher_id_3             : Identifiant avoir #3
  • Voucher_amount_3         : Montant avoir #3

[bold cyan]Correspondance avec le code C:[/bold cyan]
  Ce parser Python est équivalent à la fonction C :
    ITFWEBKIA_VENTES_decodekia()
  dans le fichier itfwebkia.ec (lignes 430-546)

[bold cyan]Références:[/bold cyan]
  • Fonction C  : ITFWEBKIA_VENTES_decodekia()
  • Structure C : ttinvoicekia
  • Fichier     : common_src/itfwebkia.ec
"""
    console.print(doc_text)


if __name__ == "__main__":
    app()
