#!/usr/bin/env python3
"""
Module de gestion de la progression enrichie via mémoire partagée.

Ce module fournit une interface Python pour communiquer avec l'interface GTK
via une structure de données enrichie en mémoire partagée.
"""

import mmap
import struct
import posix_ipc
from enum import IntEnum
from typing import Optional
import logging

logger = logging.getLogger(__name__)

SHARED_MEMORY_PREFIX = "/gcv6_etiq_shared_mem_"

def get_shared_memory_name(attached_pid: int) -> str:
    """Retourne le nom de la mémoire partagée pour un PID donné."""
    return f"{SHARED_MEMORY_PREFIX}{attached_pid}"

class EtiqGenerationStep(IntEnum):
    """Énumération des étapes de génération d'étiquettes."""
    INIT = 0
    PARSE_JSON = 1
    GENERATE_BARCODE = 2
    GENERATE_LATEX = 3
    COMPILE_PDF = 4
    FINALIZE = 5
    COUNT = 6

class EtiqSharedProgress:
    """
    Gestionnaire de la progression enrichie via mémoire partagée.

    Cette classe permet de communiquer des informations détaillées de progression
    à l'interface GTK via une structure partagée.

    Structure C correspondante:
    typedef struct {
        int progress_percent;      // 0-100
        int current_step;         // Étape actuelle (0-based)
        int total_steps;          // Nombre total d'étapes
        char step_name[64];       // Nom de l'étape actuelle
        char error_message[256];  // Message d'erreur si échec
        int status;              // 0=running, 1=success, -1=error
    } etiq_shared_progress_t;
    """

    # Taille de la structure C: 4+4+4+64+256+4 = 336 bytes
    STRUCT_SIZE = 336
    STEP_NAME_SIZE = 64
    ERROR_MESSAGE_SIZE = 256

    # Format struct pour pack/unpack: int, int, int, 64s, 256s, int
    STRUCT_FORMAT = "iii64s256si"

    def __init__(self, attached_pid: int):
        """
        Initialise le gestionnaire de progression.

        Args:
            attached_pid: PID du processus attaché pour nommer la mémoire partagée
        """
        self.attached_pid = attached_pid
        self.shm_name = get_shared_memory_name(attached_pid)
        self.memory: Optional[posix_ipc.SharedMemory] = None
        self.mm: Optional[mmap.mmap] = None

        # État interne
        self.current_step = EtiqGenerationStep.INIT
        self.total_steps = EtiqGenerationStep.COUNT
        self.progress_percent = 0
        self.status = 0  # 0=running, 1=success, -1=error

        self._initialize_shared_memory()

    def _initialize_shared_memory(self):
        """Initialise la mémoire partagée enrichie."""
        try:
            # Créer la mémoire partagée avec la nouvelle taille
            # Tente d'ouvrir la mémoire partagée existante, sinon la crée
            try:
                self.memory = posix_ipc.SharedMemory(self.shm_name)
                logger.info(f"Mémoire partagée existante ouverte: {self.shm_name}")
            except posix_ipc.ExistentialError:
                self.memory = posix_ipc.SharedMemory(
                    self.shm_name,
                    posix_ipc.O_CREAT,
                    size=self.STRUCT_SIZE
                )
                logger.info(f"Mémoire partagée créée: {self.shm_name}")
            self.mm = mmap.mmap(self.memory.fd, self.memory.size)
            self.memory.close_fd()

            # Initialiser la structure
            self._update_shared_memory(
                progress_percent=0,
                current_step=EtiqGenerationStep.INIT,
                step_name="Initialisation",
                error_message="",
                status=0
            )

            logger.debug(f"Mémoire partagée enrichie initialisée: {self.shm_name}")

        except posix_ipc.ExistentialError:
            logger.error(f"Mémoire partagée {self.shm_name} existe déjà")
            raise
        except Exception as e:
            logger.error(f"Erreur lors de l'initialisation de la mémoire partagée: {e}")
            # Fallback vers l'ancien format (4 bytes)
            self._initialize_legacy_memory()

    def _initialize_legacy_memory(self):
        """Fallback vers l'ancien format de mémoire partagée (4 bytes)."""
        try:
            self.memory = posix_ipc.SharedMemory(
                self.shm_name,
                posix_ipc.O_CREAT,
                size=4
            )
            self.mm = mmap.mmap(self.memory.fd, self.memory.size)
            self.memory.close_fd()

            # Initialiser à 0
            self.mm.seek(0)
            self.mm.write(struct.pack('i', 0))
            self.mm.flush()

            logger.warning("Fallback vers l'ancien format de mémoire partagée (4 bytes)")

        except Exception as e:
            logger.error(f"Erreur lors de l'initialisation de la mémoire partagée legacy: {e}")
            self.memory = None
            self.mm = None

    def _update_shared_memory(self, progress_percent: int, current_step: EtiqGenerationStep,
                             step_name: str, error_message: str = "", status: int = 0):
        """Met à jour la structure en mémoire partagée."""
        if not self.mm:
            return

        try:

            step_name_bytes = step_name.encode('utf-8')[:self.STEP_NAME_SIZE-1]
            step_name_padded = step_name_bytes.ljust(self.STEP_NAME_SIZE, b'\0')

            error_message_bytes = error_message.encode('utf-8')[:self.ERROR_MESSAGE_SIZE-1]
            error_message_padded = error_message_bytes.ljust(self.ERROR_MESSAGE_SIZE, b'\0')

            data = struct.pack(
                self.STRUCT_FORMAT,
                progress_percent,
                current_step,
                self.total_steps,
                step_name_padded,
                error_message_padded,
                status
            )

            self.mm.seek(0)
            self.mm.write(data)
            self.mm.flush()

        except Exception as e:
            logger.error(f"Erreur lors de la mise à jour de la mémoire partagée: {e}")

    def update_progress(self, step: EtiqGenerationStep, progress_percent: int, step_name: str):
        """
        Met à jour la progression.

        Args:
            step: Étape actuelle
            progress_percent: Pourcentage de progression (0-100)
            step_name: Nom lisible de l'étape
        """
        self.current_step = step
        self.progress_percent = progress_percent

        self._update_shared_memory(
            progress_percent=progress_percent,
            current_step=step,
            step_name=step_name,
            status=self.status
        )

    def set_error(self, error_message: str):
        """
        Signale une erreur.

        Args:
            error_message: Message d'erreur
        """
        self.status = -1

        self._update_shared_memory(
            progress_percent=self.progress_percent,
            current_step=self.current_step,
            step_name=f"Erreur: {error_message}",
            error_message=error_message,
            status=-1
        )

        logger.error(f"Erreur signalée: {error_message}")

    def set_success(self, message: str = "Génération réussie"):
        """Signale la réussite de la génération."""
        self.status = 1
        self.progress_percent = 100

        self._update_shared_memory(
            progress_percent=100,
            current_step=EtiqGenerationStep.FINALIZE,
            step_name=message,
            status=1
        )

        logger.info("Génération terminée avec succès")

    def cleanup(self):
        """Nettoie les ressources."""
        if self.mm:
            self.mm.close()
        if self.memory:
            try:
                self.memory.unlink()
            except:
                pass  # Ignore si déjà supprimé

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_type:
            self.set_error(f"Exception: {exc_val}")
        else:
            self.set_success()
        self.cleanup()

    def start_step(self, step: EtiqGenerationStep):
        """Démarre une étape de progression."""
        self.current_step = step
        self.update_progress(step, 0, get_step_name(step))
        logger.info(f"Étape {get_step_name(step)} démarrée")

    def end_current_step(self):
        """Termine une étape de progression."""
        if self.current_step == EtiqGenerationStep.FINALIZE:
            self.set_success()
        else:
            self.update_progress(self.current_step, 100, get_step_name(self.current_step))
        logger.info(f"Étape {get_step_name(self.current_step)} terminée")

    def update_step_progress(self, progress_percent: int, message: str = ""):
        """Met à jour la progression de l'étape actuelle."""
        if self.current_step is not None:
            self.update_progress(self.current_step, progress_percent, message)
            logger.info(f"Progression de l'étape {get_step_name(self.current_step)} mise à jour à {progress_percent}%")
        else:
            logger.warning("Aucune étape en cours pour mettre à jour la progression")

    def update_step_by_count(self, count: int, total: int, message: str = ""):
        """Met à jour la progression de l'étape actuelle en fonction du nombre d'éléments traités."""
        if total <= 0:
            logger.error("Total doit être supérieur à 0 pour mettre à jour la progression")
            return
        if count < 0 or count > total:
            logger.error(f"Count {count} invalide pour le total {total}")
            return

        progress_percent = int((count / total) * 100)
        self.update_progress(self.current_step, progress_percent, message)

# Noms des étapes en français
STEP_NAMES = {
    EtiqGenerationStep.INIT: "Initialisation",
    EtiqGenerationStep.PARSE_JSON: "Analyse des données",
    EtiqGenerationStep.GENERATE_LATEX: "Génération ...",
    EtiqGenerationStep.COMPILE_PDF: "Génération PDF",
    EtiqGenerationStep.FINALIZE: "Finalisation"
}

def get_step_name(step: EtiqGenerationStep) -> str:
    """Retourne le nom français d'une étape."""
    return STEP_NAMES.get(step, f"Étape {step}")



def exemple_usage():
    """Exemple d'utilisation de la classe EtiqSharedProgress."""
    pid = 2207  # Exemple de PID
    progress_manager = EtiqSharedProgress(pid)

    progress_manager.update_progress(EtiqGenerationStep.INIT, 10, "Initialisation")
    progress_manager.set_error("Une erreur s'est produite pendant l'initialisation")
    progress_manager.set_success()
    progress_manager.cleanup()

    print(f"Mémoire partagée {get_shared_memory_name(pid)} nettoyée.")
