#!/usr/bin/env python3
"""
Module de configuration et validation pour le système d'étiquettes.

Ce module vérifie et configure automatiquement l'environnement nécessaire
au bon fonctionnement du système de génération d'étiquettes.
"""

import logging
import shutil
import subprocess
import sys
from pathlib import Path
from typing import Dict, List, Optional, Tuple
import typer

logger = logging.getLogger(__name__)

# Configuration des chemins système
SYSTEM_PATHS = {
    "etiquettes_root": Path("/home/local/etc/gcv6/etiquettes"),
    "design_dir": Path("/home/local/etc/gcv6/etiquettes/design"),
    "fonts_dir": Path("/home/local/fonts"),
    "texlive_bin": Path("/home/local/texlive/2025/bin/x86_64-linux"),
}

# Dépendances système requises
SYSTEM_DEPENDENCIES = {
    "lualatex": "LaTeX avec support LuaTeX",
    "convert": "ImageMagick pour conversion PDF vers PNG",
    "pkg-config": "Configuration des bibliothèques système",
}

# Modules Python requis
PYTHON_DEPENDENCIES = [
    "typer",
    "pydantic", 
    "structlog",
    "tqdm",
    "pandas",
    "numpy",
    "posix_ipc",
    "requests",
    "PyYAML",
]

def check_command_available(command: str) -> bool:
    """
    Vérifie si une commande système est disponible.
    
    Args:
        command: Nom de la commande à vérifier
        
    Returns:
        True si la commande est disponible, False sinon
    """
    return shutil.which(command) is not None

def check_python_module(module_name: str) -> bool:
    """
    Vérifie si un module Python est disponible.
    
    Args:
        module_name: Nom du module à vérifier
        
    Returns:
        True si le module est disponible, False sinon
    """
    try:
        __import__(module_name.replace("-", "_"))
        return True
    except ImportError:
        return False

def check_directory_writable(path: Path) -> bool:
    """
    Vérifie si un répertoire existe et est accessible en écriture.
    
    Args:
        path: Chemin du répertoire à vérifier
        
    Returns:
        True si le répertoire est accessible en écriture, False sinon
    """
    try:
        if not path.exists():
            return False
        # Test d'écriture avec un fichier temporaire
        test_file = path / ".write_test"
        test_file.touch()
        test_file.unlink()
        return True
    except (OSError, PermissionError):
        return False

def create_directory_structure() -> Dict[str, bool]:
    """
    Crée la structure de répertoires nécessaire pour les étiquettes.
    
    Returns:
        Dictionnaire avec les résultats de création pour chaque répertoire
    """
    results = {}
    
    for name, path in SYSTEM_PATHS.items():
        try:
            path.mkdir(parents=True, exist_ok=True)
            
            # Créer les sous-répertoires spécifiques pour design_dir
            if name == "design_dir":
                # Créer les répertoires images pour chaque design existant
                from .config import get_designs
                try:
                    designs = get_designs()
                    for design_name in designs.keys():
                        images_dir = path / design_name / "images"
                        images_dir.mkdir(parents=True, exist_ok=True)
                        logger.info(f"Répertoire d'images créé: {images_dir}")
                except Exception as e:
                    logger.warning(f"Impossible de créer les répertoires de design: {e}")
            
            results[name] = True
            logger.info(f"Répertoire créé/vérifié: {path}")
            
        except (OSError, PermissionError) as e:
            results[name] = False
            logger.error(f"Impossible de créer le répertoire {path}: {e}")
            
    return results

def check_latex_installation() -> Tuple[bool, Optional[str]]:
    """
    Vérifie l'installation LaTeX et ses capacités.
    
    Returns:
        Tuple (installation_ok, version_info)
    """
    try:
        # Vérifier LuaLaTeX
        result = subprocess.run(
            ["lualatex", "--version"], 
            capture_output=True, 
            text=True, 
            timeout=10
        )
        
        if result.returncode == 0:
            version_line = result.stdout.split('\n')[0]
            logger.info(f"LaTeX trouvé: {version_line}")
            return True, version_line
        else:
            return False, f"Erreur lualatex: {result.stderr}"
            
    except (subprocess.TimeoutExpired, FileNotFoundError) as e:
        return False, f"LuaLaTeX non disponible: {e}"

def check_imagemagick_installation() -> Tuple[bool, Optional[str]]:
    """
    Vérifie l'installation ImageMagick.
    
    Returns:
        Tuple (installation_ok, version_info)
    """
    try:
        result = subprocess.run(
            ["convert", "-version"], 
            capture_output=True, 
            text=True, 
            timeout=10
        )
        
        if result.returncode == 0:
            version_line = result.stdout.split('\n')[0]
            logger.info(f"ImageMagick trouvé: {version_line}")
            return True, version_line
        else:
            return False, f"Erreur convert: {result.stderr}"
            
    except (subprocess.TimeoutExpired, FileNotFoundError) as e:
        return False, f"ImageMagick non disponible: {e}"

def validate_environment() -> Dict[str, Dict[str, any]]:
    """
    Effectue une validation complète de l'environnement.
    
    Returns:
        Dictionnaire détaillé des résultats de validation
    """
    results = {
        "system_dependencies": {},
        "python_dependencies": {},
        "directories": {},
        "latex": {},
        "imagemagick": {},
        "overall_status": True
    }
    
    # Vérification des dépendances système
    logger.info("Vérification des dépendances système...")
    for cmd, description in SYSTEM_DEPENDENCIES.items():
        available = check_command_available(cmd)
        results["system_dependencies"][cmd] = {
            "available": available,
            "description": description
        }
        if not available:
            results["overall_status"] = False
            logger.error(f"Dépendance manquante: {cmd} ({description})")
    
    # Vérification des modules Python
    logger.info("Vérification des modules Python...")
    for module in PYTHON_DEPENDENCIES:
        available = check_python_module(module)
        results["python_dependencies"][module] = {"available": available}
        if not available:
            results["overall_status"] = False
            logger.error(f"Module Python manquant: {module}")
    
    # Vérification des répertoires
    logger.info("Vérification des répertoires...")
    for name, path in SYSTEM_PATHS.items():
        exists = path.exists()
        writable = check_directory_writable(path) if exists else False
        results["directories"][name] = {
            "path": str(path),
            "exists": exists,
            "writable": writable
        }
        if not exists or not writable:
            results["overall_status"] = False
            if not exists:
                logger.warning(f"Répertoire manquant: {path}")
            else:
                logger.error(f"Répertoire non accessible en écriture: {path}")
    
    # Vérification spéciale LaTeX
    latex_ok, latex_info = check_latex_installation()
    results["latex"] = {
        "available": latex_ok,
        "info": latex_info
    }
    if not latex_ok:
        results["overall_status"] = False
    
    # Vérification spéciale ImageMagick
    im_ok, im_info = check_imagemagick_installation()
    results["imagemagick"] = {
        "available": im_ok,
        "info": im_info
    }
    if not im_ok:
        results["overall_status"] = False
    
    return results

def print_validation_report(results: Dict[str, Dict[str, any]]) -> None:
    """
    Affiche un rapport détaillé de validation.
    
    Args:
        results: Résultats de la validation
    """
    print("\n" + "="*80)
    print("RAPPORT DE VALIDATION DU SYSTÈME D'ÉTIQUETTES")
    print("="*80)
    
    # Statut global
    status_symbol = "✅" if results["overall_status"] else "❌"
    print(f"\nStatut global: {status_symbol}")
    
    # Dépendances système
    print(f"\n📦 DÉPENDANCES SYSTÈME:")
    for cmd, info in results["system_dependencies"].items():
        symbol = "✅" if info["available"] else "❌"
        print(f"  {symbol} {cmd}: {info['description']}")
    
    # Modules Python
    print(f"\n🐍 MODULES PYTHON:")
    for module, info in results["python_dependencies"].items():
        symbol = "✅" if info["available"] else "❌"
        print(f"  {symbol} {module}")
    
    # Répertoires
    print(f"\n📁 RÉPERTOIRES:")
    for name, info in results["directories"].items():
        if info["exists"] and info["writable"]:
            symbol = "✅"
            status = "OK"
        elif info["exists"]:
            symbol = "⚠️"
            status = "Lecture seule"
        else:
            symbol = "❌"
            status = "Manquant"
        print(f"  {symbol} {name}: {info['path']} ({status})")
    
    # LaTeX
    print(f"\n📄 LATEX:")
    latex_symbol = "✅" if results["latex"]["available"] else "❌"
    print(f"  {latex_symbol} LuaLaTeX: {results['latex']['info']}")
    
    # ImageMagick
    print(f"\n🖼️  IMAGEMAGICK:")
    im_symbol = "✅" if results["imagemagick"]["available"] else "❌"
    print(f"  {im_symbol} Convert: {results['imagemagick']['info']}")
    
    # Recommandations
    if not results["overall_status"]:
        print(f"\n💡 RECOMMANDATIONS:")
        
        # Dépendances système manquantes
        missing_deps = [cmd for cmd, info in results["system_dependencies"].items() 
                       if not info["available"]]
        if missing_deps:
            print(f"  • Installer les dépendances système: {', '.join(missing_deps)}")
            print(f"    sudo dnf install ImageMagick")
        
        # Modules Python manquants
        missing_modules = [mod for mod, info in results["python_dependencies"].items() 
                          if not info["available"]]
        if missing_modules:
            print(f"  • Installer les modules Python: pip install {' '.join(missing_modules)}")
        
        # Répertoires manquants
        missing_dirs = [name for name, info in results["directories"].items() 
                       if not info["exists"]]
        if missing_dirs:
            print(f"  • Créer les répertoires manquants avec: gescokit labels setup")
    
    print("\n" + "="*80 + "\n")

# Interface CLI avec typer
app = typer.Typer(help="Configuration et validation du système d'étiquettes")

@app.command("check")
def check_environment(
    verbose: bool = typer.Option(False, "--verbose", "-v", help="Affichage détaillé")
):
    """
    Vérifie l'environnement et les dépendances du système d'étiquettes.
    """
    # Configuration du logging
    log_level = logging.DEBUG if verbose else logging.INFO
    logging.basicConfig(level=log_level, format='%(levelname)s: %(message)s')
    
    logger.info("Démarrage de la validation de l'environnement...")
    
    # Effectuer la validation
    results = validate_environment()
    
    # Afficher le rapport
    print_validation_report(results)
    
    # Code de sortie
    if results["overall_status"]:
        typer.echo("✅ Environnement valide pour la génération d'étiquettes")
        raise typer.Exit(0)
    else:
        typer.echo("❌ Problèmes détectés dans l'environnement", err=True)
        raise typer.Exit(1)

@app.command("setup")
def setup_environment(
    force: bool = typer.Option(False, "--force", "-f", help="Forcer la création même si les répertoires existent"),
    verbose: bool = typer.Option(False, "--verbose", "-v", help="Affichage détaillé")
):
    """
    Configure l'environnement en créant les répertoires nécessaires.
    """
    # Configuration du logging
    log_level = logging.DEBUG if verbose else logging.INFO
    logging.basicConfig(level=log_level, format='%(levelname)s: %(message)s')
    
    logger.info("Configuration de l'environnement pour les étiquettes...")
    
    # Créer la structure de répertoires
    results = create_directory_structure()
    
    # Afficher les résultats
    success_count = sum(1 for success in results.values() if success)
    total_count = len(results)
    
    typer.echo(f"\nCréation des répertoires: {success_count}/{total_count} réussis")
    
    for name, success in results.items():
        symbol = "✅" if success else "❌"
        path = SYSTEM_PATHS[name]
        typer.echo(f"  {symbol} {name}: {path}")
    
    if success_count == total_count:
        typer.echo("\n✅ Configuration terminée avec succès")
        typer.echo("Vous pouvez maintenant générer les aperçus avec:")
        typer.echo("  gescokit labels generate-previews")
    else:
        typer.echo(f"\n⚠️ Certains répertoires n'ont pas pu être créés", err=True)
        typer.echo("Vérifiez les permissions et réessayez avec des droits administrateur si nécessaire")
        raise typer.Exit(1)

if __name__ == "__main__":
    app()