import logging
from pathlib import Path
import os
import re
import sys
from typing import Tuple

logger = logging.getLogger(__name__)

from yaml import load, Loader

def load_config(config_file, encoding="utf8"):
    with open( str(config_file), encoding=encoding  ) as cfg:
        data = load(cfg, Loader=Loader)

    return data


def get_home():
    if hasattr(Path, "home"):
        return Path.home()
    else:
        return Path(os.path.expanduser("~"))

def get_centos_release():
    if Path("/etc/centos-release").is_file():
        with open("/etc/centos-release") as fp:
            line = fp.readline()
            matched = re.search("(\d)", line.strip())
            if matched:
                return int(matched.group(0))
    print("etiquettes: Impossible d'obtenir la version de centos. /etc/centos-release n'est pas un fichier")
    return None

def get_design_data(user_config, display_nom):
    """Retourne le modele correspondant au display_nom
    Returns
    -------
        modelePy: str
             le nom de classe Python
        modele: dict
             la configuration du modèle (couleurs, fonts...)
    """
    for design_data in user_config.values():
        display_name = design_data.get("display_nom")
        if display_name == display_nom:
            return design_data
    raise ValueError("Modele [{0}] introuvable dans les fichiers de configuration de la société.".format(display_nom))

def get_design_shortname(user_config, display_nom):
    """Retourne le nom court du modele/design correspondant au display_nom (nom long)
    Returns
    -------
        shortname: str
             le nom court du modèle/design
    """
    for shortname, design_data in user_config.items():
        display_name = design_data.get("display_nom")
        if display_name == display_nom:
            return shortname
    raise ValueError("Modele [{0}] introuvable dans les fichiers de configuration de la société.".format(display_nom))

def get_formats_dispo(config):
    formats = list(config)
    formats.remove("display_nom")
    return formats

def enlever_parenthese(txt:str):
    matched = re.match("^\((.*)\)$", txt)
    if not matched:
        return txt
    return matched.group(1)

def isPython34():
    return (sys.version_info[0] == 3) and (sys.version_info[1] == 4)

def assert_python3():
    if sys.version_info[0] < 3:
        raise ValueError("Version de python incompatible", sys.version_info[0])


