import functools
import math

@functools.total_ordering
class Prix:
    """Gestion du prix pour impression sur etiquettes.

    Définit la longueur du prix, qui est le nombre de chiffres dans la partie entière.
    Stocke l’éco-participation et son unité, le colisage et son unité.
    Sépare le prix en sa partie décimale et sa partie centimes.
    """
    __unit_mapping = {
        "L": [1,"L"],
        "ml": [1000,"L"],
        "cl": [100,"L"],

        "kg": [1,"kg"],
        "g": [1000,"kg"],

    }

    def __init__(self, prix, ecopart=None, decimal_mark=",",
                 contenance=None, contenance_unit=None,
                 ecomeuble=None,
                 colisage=None, colisage_unit=None, *args, **kwargs):
        self.prix = prix or 0
        self.ecopart = ecopart or None #"dont écopart : 0,00€ / 0L/kg, soit 0,00€ au L/kg"
        self.decimal_mark = decimal_mark or ","
        self.contenance = contenance or None
        self.contenance_unit = contenance_unit or None
        self.ecomeuble = ecomeuble or None
        self.colisage = colisage or None
        self.colisage_unit = colisage_unit or None

        self.entier, self.centimes = self._split()
        # print("etiquettes/chore.py Pk3 autant d'apparition ?cc")
        self.contenance_prix = self._contenance_prix()

    def _contenance_prix(self):
        """Calcul le prix au L, kg..."""
        if not self.contenance:
            return None
        factor, unit = self.__unit_mapping.get(self.contenance_unit, [1, self.contenance_unit])

        # return "{:.02f}".format(self.prix/self.contenance)
        prix_a_la_contenance = self.prix/self.contenance
        if factor == 1:
            rslt = "{:.02f}".format(prix_a_la_contenance).replace(".", self.decimal_mark)
        else:
            self.contenance_unit = unit
            rslt = "{:.02f}".format(prix_a_la_contenance*factor).replace(".", self.decimal_mark)
        # Supprime les 0 si le prix est un entier
        if rslt.endswith(f"{self.decimal_mark}00"):
            rslt = rslt[:-3]
        return rslt

    def _split(self):
        """Retourne 2 chaines de caractères. La première contient la valeur entière du prix, la seconde les centimes
        Returns
        -------
            tuple: (entier, centimes)
        """
        return [int(val) for val in "{:.02f}".format(self.prix).split(".")]

    def __str__(self)->str:
        """Affiche le prix en str"""
        return f"{self.entier}{self.decimal_mark}{self.centimes}"

    def __len__(self)->int:
        """Nombre de chiffres composant la valeur entière"""
        if self.entier == 0:
            return 1
        return math.floor(math.log10(abs(self.entier))) + 1

    def _is_valid_operand(self, other):
        return isinstance(other, Prix) or isinstance(other, int) or isinstance(other, float)

    def __lt__(self, other):
        if not self._is_valid_operand(other):
            return NotImplemented

        if isinstance(other, Prix):
            return self.prix < other.prix

        return self.prix < other

    def __eq__(self, other):
        if not self._is_valid_operand(other):
            return NotImplemented
        if isinstance(other, Prix):
            return self.prix == other.prix

        return self.prix == other

    def to_dict(self):
        """Retourne un dictionnaire contenant les attributs de la classe"""
        return {key: getattr(self, key) for key in dir(self) if not key.startswith("_") and not callable(getattr(self, key))}


class PrixHtTtc:
    """Sauvegarde le prix en HT et en TTC voir Prix"""
    def __init__(self, ht, ttc, *args, **kwargs):
        self.ht  = Prix(ht, *args, **kwargs)
        self.ttc = Prix(ttc, *args, **kwargs)

    def __str__(self) -> str:
        return "HT {0} TTC {1}".format( self.ht, self.ttc)

    def to_dict(self):
        """Retourne un dictionnaire contenant les attributs de la classe"""
        return {"ht": self.ht.to_dict(), "ttc": self.ttc.to_dict()}
