from collections import OrderedDict
import functools
import itertools
import logging
import re

from pypdf import PdfReader

from typing import Iterator, Tuple, Dict, Any

logger = logging.getLogger(__name__)

def auto_detection_config_papier(articles: list[dict], paperSize: str) -> Dict[str, Any]:
    """
    Détermine si toutes les étiquettes ont le même format que la taille du papier.

    Parameters
    ----------
    articles : list of dict
        Liste des articles à imprimer.
    paperSize : str
        Taille du papier (ex: A4, A7).

    Returns
    -------
    dict
        Dictionnaire contenant :
        - paper_equal_format : bool
            True si le format est identique à la taille du papier.
    """

    config_papier = {"paper_equal_format": False}

    # is_landscape = []
    # for eti in articles:
    #     if "papiers_paysage" in eti.config and args.paper in eti.config["papiers_paysage"]:
    #         is_landscape.append( eti.config["papiers_paysage"][args.paper] )
    #     else:
    #         is_landscape.append(False)

    # is_landscape = list( set(is_landscape) )
    # if len(is_landscape) == 1:
    #     config_papier["landscape"] = is_landscape[0]

    formats = {eti["format"] for eti in articles}
    if len(formats) == 1:
        # TODO: Je suppose que toutes les étiquettes seront du même format
        if paperSize == list(formats)[0]:
            config_papier["paper_equal_format"] = True
            # config_papier["landscape"] = etiquettes.PaperSize(args.paper).landscape
    return config_papier



#======================================================================================
# Déterminer l'organisation des étiquettes dans une page puis dans le document complet
#======================================================================================
def str_to_mm(txt: str) -> Tuple[float, str | None]:
    """
    Convertit une chaîne de type '7cm' ou '30mm' en valeur millimétrique.

    Parameters
    ----------
    txt : str
        La chaîne à convertir.

    Returns
    -------
    float
        La valeur en millimètres.
    str | None
        L'unité utilisée, ou None si non reconnue.
    """
    matched = re.search("^(\d*\.?\d+)(cm|mm)$", txt)

    if not matched:
        return 0, None
    valeur = float(matched.group(1))
    unit = matched.group(2)
    if unit == "cm":
        valeur *= 10
        unit = "mm"
    return valeur, unit

def calcul_positions_etiquettes(
        paperSize: str,
        fmt: str,
        margins: dict[str, str],
        landscape: bool,
        nbetiquettes: int
) -> Tuple[Iterator[str], int]:

    """
    Calcule la position de chaque étiquette sur la page et le nombre total par page.

    Parameters
    ----------
    papier: str
        taille, format du papier (à l'imprimante): a4, a7, gondole...
    fmt: str
        format des étiquettes. On suppose qu'il n'y a qu'un format à imprimer, c-a-d
        toutes les étiquettes ont le même format
    margins : dict
        Marges à utiliser pour le placement
    landscape : bool
        Orientation paysage ou non
    nbetiquettes : int
        Nombre total d’étiquettes à générer

    Returns
    -------
    Tuple[Iterator[str], int]
        Un itérateur infini de positions (x, y) et le nombre d'étiquettes par page.
    """


    if fmt == paperSize:
        return itertools.cycle(['(0.0mm,0.0mm)']), 1

    left, top, hspace, vspace = 0, 0, 0, 0
    if margins:
        left, _ = str_to_mm(margins.get("left", "0mm"))
        top, _  = str_to_mm(margins.get("top", "0mm"))
        left *= 2 # Suppose marges gauche = droite
        top *= 2  # Suppose marges haut = bas

        hspace, _ = str_to_mm(margins.get("eti_right","0mm"))
        vspace, _ = str_to_mm(margins.get("eti_bot","0mm"))

    papier_width, papier_height = PaperSize(paperSize).dimension
    if landscape:
        papier_width, papier_height = papier_height, papier_width

    # Supprimer les marges
    papier_height -= top
    papier_width -= left
    # Déterminer la distribution des étiquettes sur une page
    w,h = PaperSize(fmt).dimension
    w += hspace
    h += vspace
    ncols, nrows = int(papier_width//w), int(papier_height//h)

    nbetiquettes_par_page = ncols*nrows

    if nbetiquettes_par_page == 0:
        raise ValueError("Impossible de placer une étiquette sur la page. Vérifier les marges et la taille des étiquettes")

    nbpages, nbetiq_restantes = divmod(nbetiquettes, nbetiquettes_par_page )
    if nbetiq_restantes > 0:
        nbpages += 1

    logger.info(f"{nbetiquettes} étiquettes réparties sur {nbpages} page(s)")
    logger.info(f"Disposition : {ncols} colonne(s) x {nrows} ligne(s) par page")

    eti_positions_une_page = [f"({i * w}mm,-{j * h}mm)" for j in range(nrows) for i in range(ncols)]

    return itertools.cycle(eti_positions_une_page), nbetiquettes_par_page

def get_page_size_from_pdf(pdf_file: str) -> Tuple[float, float]:
    """
    Obtient la taille de la première page d'un fichier PDF.

    Parameters
    ----------
    pdf_file : str
        Chemin vers le fichier PDF.

    Returns
    -------
    Tuple[float, float]
        Largeur et hauteur de la première page en millimètres.
    """
    reader = PdfReader(pdf_file)
    page = reader.pages[0]
    # pypdf donne les dimensions en points (1/72 pouce)
    # Convertir en millimètres: points * 25.4 / 72
    width = float(page.mediabox.width) * 25.4 / 72
    height = float(page.mediabox.height) * 25.4 / 72
    return width, height

def get_nb_pages_from_pdf(pdf_file: str) -> int:
    """
    Obtient le nombre de pages d'un fichier PDF.

    Parameters
    ----------
    pdf_file : str
        Chemin vers le fichier PDF.

    Returns
    -------
    int
        Nombre de pages dans le PDF.
    """
    reader = PdfReader(pdf_file)
    return len(reader.pages)

def get_nbPDFpages(texlogfile):
    with open(texlogfile) as fp:
        lines = fp.readlines()[::-1]
        while lines:
            line = lines.pop(0).strip()
            if line.startswith("Output written on"):
                matched = re.search("(\d+) page", line)
                if matched:
                    return matched.group(1)
    logger.warning("Impossible de déterminer le nombre de pages du PDF")
    return 0


@functools.total_ordering
class PaperSize:
    """Gestion des tailles de papier pour pouvoir les comparer et s’assurer qu’une étiquette peut
    s’afficher dans le format papier"""

    # Toujours trier les papiers dans l'ordre croissant (est-ce que ce papier passe dans le précédent?)
    # Toutes les dimensions sont exprimées en mm
    _papersizes = OrderedDict(
        (
        ("gondole", "52.5mm:25mm"),
        ("a7"     , "105mm:74mm"),
        ("15x15"  , "150mm:150mm"),
        ("a6"     , "105mm:148mm"),
        ("a5"     , "148mm:210mm"),
        ("a4"     , "210mm:297mm"),
        ("a3"     , "297mm:420mm"),
        ("a2"     , "420mm:594mm"),
        ("a1"     , "594mm:841mm"),
        ("a0"     , "841mm:1189mm")
        ))


    def __init__(self, paper, landscape=False):
        if paper not in self._papersizes:
            raise ValueError("Impossible de trouver le papier {}. Veuillez éditer PaperSize".format(paper))
        self.paper = paper
        matched = re.search("^(\d*\.?\d+)mm:(\d*\.?\d+)mm$", self._papersizes[self.paper])

        if matched:
            self.width = float(matched.group(1))
            self.height = float(matched.group(2))
        else:
            raise ValueError("Le formatage accepté est: 'nom', '10mm:20mm' au lieu de {1} pour {0}".format(self.paper, self._papersizes[self.paper]))

        self.landscape = self.height < self.width

    def as_scrartcl_opt(self) -> str:
        return self.paper if self.paper.startswith("a") else self._papersizes[self.paper]

    def __str__(self):
        return self.paper

    def info(self):
        return "{} {}x{}".format(self.paper, self.width, self.height)

    @property
    def dimension(self):
        return self.width, self.height

    def _is_valid_operand(self, other):
        return hasattr(other, "paper")

    def __pos(self):
        """Position du papier dans la liste _papers"""
        return list(self._papersizes.keys()).index(self.paper)

    def __lt__(self, other):
        if not self._is_valid_operand(other):
            return NotImplemented

        iself = self.__pos()
        iothe = other.__pos()
        return iself < iothe

    def __eq__(self, other):
        if not self._is_valid_operand(other):
            return NotImplemented
        iself = self.__pos()
        iothe = other.__pos()
        return iself == iothe