#!/usr/bin/env python3
"""
Module pour générer des exemples d'étiquettes montrant différentes topologies.

Ce module génère un PDF au format A4 (ou autre) contenant plusieurs exemples
d'étiquettes avec différentes topologies (prix, promotions, etc.) pour un format donné.
"""

import json
import logging
import tempfile
from pathlib import Path
from typing import Optional, List, Dict, Any

from .config import get_designs, get_design_config
from .main import generate_labels_from_json

logger = logging.getLogger(__name__)

def create_example_articles(format_name: str) -> List[Dict[str, Any]]:
    """
    Crée une liste d'articles d'exemple avec différentes topologies.
    
    Args:
        format_name: Le nom du format d'étiquette
        
    Returns:
        Liste d'articles avec différentes configurations
    """
    base_article = {
        "categorie": format_name,
        "format": format_name,
        "nombex": 1,
        "barcode": "1234567890123",
        "lettrage": "A",
        "contenance": 200,
        "unite": "ml",
    }
    
    examples = []
    
    # Exemple 1: Article normal (PUB)
    examples.append({
        **base_article,
        "code": "EXAMPLE-001",
        "designation": "Article normal - Prix standard",
        "designationCom": "Article normal - Prix standard",
        "categorie": "PUB",
        "prix": {
            "ht": 10.94009,
            "ttc": 11.87,
            "deee": 0,
            "ecomeuble": 0
        }
    })
    
    # Exemple 2: Article en promotion (PROMO)
    examples.append({
        **base_article,
        "code": "EXAMPLE-002",
        "designation": "Article en promotion - Prix barré",
        "designationCom": "Article en promotion - Prix barré",
        "categorie": "PROMO",
        "prix": {
            "ht": 10.94009,
            "ttc": 11.87,
            "deee": 0,
            "ecomeuble": 0
        },
        "prix_ancien": {
            "ht": 15.6221,
            "ttc": 16.95
        }
    })
    
    # Exemple 3: Article VU À LA TV
    examples.append({
        **base_article,
        "code": "EXAMPLE-003",
        "designation": "Article VU À LA TV",
        "designationCom": "Article VU À LA TV",
        "categorie": "VUALATV",
        "prix": {
            "ht": 8.76,
            "ttc": 9.50,
            "deee": 0,
            "ecomeuble": 0
        },
        "promo_tarif": {
            "gratuit": 0,
            "lot": 3,
            "lot_prix": 25.00,
            "lot_prix_unitaire": 8.33
        }
    })
    
    # Exemple 4: Article avec prix élevé (À SAISIR)
    examples.append({
        **base_article,
        "code": "EXAMPLE-004",
        "designation": "Article prix élevé - 4 chiffres",
        "designationCom": "Article prix élevé - 4 chiffres",
        "categorie": "ASAISIR",
        "prix": {
            "ht": 912.15,
            "ttc": 999.99,
            "deee": 0,
            "ecomeuble": 0
        }
    })
    
    # Exemple 5: Article avec longue désignation (OPE)
    examples.append({
        **base_article,
        "code": "EXAMPLE-005",
        "designation": "Article avec une très longue désignation qui doit être tronquée selon les règles du format",
        "designationCom": "Article avec une très longue désignation qui doit être tronquée selon les règles du format",
        "categorie": "OPE",
        "prix": {
            "ht": 45.67,
            "ttc": 49.50,
            "deee": 0,
            "ecomeuble": 0
        }
    })
    
    # Exemple 6: Article avec DEEE/EcoMeuble (REMB)
    examples.append({
        **base_article,
        "code": "EXAMPLE-006",
        "designation": "Article avec DEEE et EcoMeuble",
        "designationCom": "Article avec DEEE et EcoMeuble",
        "categorie": "REMB",
        "prix": {
            "ht": 123.45,
            "ttc": 134.00,
            "deee": 2.50,
            "ecomeuble": 1.20
        }
    })
    
    # Exemple 7: Article avec prix très petit (PUB)
    examples.append({
        **base_article,
        "code": "EXAMPLE-007",
        "designation": "Article très bon marché",
        "designationCom": "Article très bon marché",
        "categorie": "PUB",
        "prix": {
            "ht": 0.45,
            "ttc": 0.50,
            "deee": 0,
            "ecomeuble": 0
        }
    })
    
    # Exemple 8: Article gratuit (PROMO)
    examples.append({
        **base_article,
        "code": "EXAMPLE-008",
        "designation": "Article gratuit - Offre spéciale",
        "designationCom": "Article gratuit - Offre spéciale",
        "categorie": "PROMO",
        "prix": {
            "ht": 0.00,
            "ttc": 0.00,
            "deee": 0,
            "ecomeuble": 0
        },
        "promo_tarif": {
            "gratuit": 1,
            "lot": 0
        }
    })
    
    return examples

def generate_examples(
    design: Optional[str] = None,
    format: Optional[str] = None,
    output: Optional[Path] = None,
    paper: str = "a4",
    verbose: bool = False,
    no_compile: bool = False
) -> Optional[Path]:
    """
    Génère un PDF d'exemples montrant différentes topologies d'étiquettes.
    
    Args:
        design: Design spécifique à utiliser (optionnel)
        format: Format spécifique à utiliser (optionnel)
        output: Fichier PDF de sortie (optionnel)
        paper: Taille du papier (a4, a3, etc.)
        verbose: Affichage détaillé
        no_compile: Ne compile pas le fichier LaTeX
        
    Returns:
        Path vers le fichier généré ou None en cas d'erreur
    """
    try:
        # Vérifier que le design existe
        designs = get_designs()
        if not designs:
            logger.error("Aucun design disponible")
            return None
        
        # Déterminer le design à utiliser
        if design is None:
            design = list(designs.keys())[0]  # Utiliser le premier design disponible
            logger.info(f"Utilisation du design par défaut: {design}")
        
        if design not in designs:
            logger.error(f"Design '{design}' non trouvé. Designs disponibles: {list(designs.keys())}")
            return None
        
        # Obtenir la configuration du design
        config, config_file = get_design_config(design)
        if not config:
            logger.error(f"Configuration du design '{design}' non trouvée")
            return None
        
        # Déterminer le format à utiliser
        # Dans le fichier YAML, les formats sont des clés directes (gondole, a7, etc.)
        # Exclure les clés qui ne sont pas des formats (display_nom, etc.)
        available_formats = [k for k in config.keys() if k not in ["display_nom"]]
        if not available_formats:
            logger.error(f"Aucun format disponible pour le design '{design}'")
            return None
        
        if format is None:
            format = available_formats[0]  # Utiliser le premier format disponible
            logger.info(f"Utilisation du format par défaut: {format}")
        elif format not in available_formats:
            logger.error(f"Format '{format}' non trouvé pour le design '{design}'. Formats disponibles: {available_formats}")
            return None
        
        # Créer les articles d'exemple
        examples = create_example_articles(format)
        logger.info(f"Génération de {len(examples)} exemples pour le format '{format}'")
        
        # Créer le fichier JSON temporaire
        output_file = output or Path(f"/tmp/etiquettes_examples_{design}_{format}.pdf")
        
        example_data = {
            "output": str(output_file),
            "papier": paper.lower(),
            "design": design,
            "format": format,
            "articles": examples
        }
        
        # Écrire le fichier JSON temporaire
        with tempfile.NamedTemporaryFile(mode='w', prefix='eti_examples_', suffix='.json', 
                                       delete=False, encoding='utf-8') as temp_file:
            json.dump(example_data, temp_file, ensure_ascii=False, indent=2)
            temp_json_path = Path(temp_file.name)
        
        # Créer un fichier de configuration JSON temporaire pour la compatibilité
        config_data = {
            "designs": {
                design: {
                    "formats": {
                        format: {
                            "name": format,
                            "enabled": True
                        }
                    }
                }
            }
        }
        
        with tempfile.NamedTemporaryFile(mode='w', prefix='eti_config_', suffix='.json', 
                                       delete=False, encoding='utf-8') as temp_config:
            json.dump(config_data, temp_config, ensure_ascii=False, indent=2)
            temp_config_path = Path(temp_config.name)
        
        logger.info(f"Génération des exemples avec:")
        logger.info(f"  Design: {design}")
        logger.info(f"  Format: {format}")
        logger.info(f"  Papier: {paper}")
        logger.info(f"  Sortie: {output_file}")
        logger.info(f"  Nombre d'articles: {len(examples)}")
        
        # Générer le PDF
        result = generate_labels_from_json(
            json_file=temp_json_path,
            config_file=temp_config_path,
            design_shortname=design,
            output=output_file,
            paper_size=paper,
            max_labels=0,  # Pas de limite
            debug=2 if verbose else 0,
            verbose=verbose,
            no_latex=no_compile,
            build_pngs=False
        )
        
        # Nettoyer les fichiers temporaires
        try:
            temp_json_path.unlink()
            temp_config_path.unlink()
        except:
            pass
        
        if result:
            logger.info(f"Fichier d'exemples généré avec succès: {result}")
            return Path(result)
        else:
            logger.error("Échec de la génération des exemples")
            return None
            
    except Exception as e:
        logger.error(f"Erreur lors de la génération des exemples: {e}")
        return None