
import typer
from pathlib import Path
from typing import Optional

from gescokit.common.utils import check_lualatex_installed, check_arara_installed, check_pdftocairo_installed

from .main import generate_labels_from_json
from .generate_previews import generate_all_previews
from .generate_examples import generate_examples
from .setup import app as setup_app

app = typer.Typer(help="Générateur d'étiquettes à partir d'un fichier JSON")

# Ajouter les commandes de setup
app.add_typer(setup_app, name="setup", help="Configuration et validation de l'environnement")

@app.command("generate")
def generate(
    json_file: Path = typer.Option(..., "--json-file", "-j", help="Fichier JSON contenant les articles provenant de l'ERP"),
    config_file: Path = typer.Option(..., "--config", "-c", help="Indique le fichier de configuration JSON (souvent nommé modeles_etiquette.json) qui contient les designs et les configurations des étiquettes."),
    design: Optional[str] = typer.Option(None, "--design", "-d", help="Nom (short name) du design à utiliser pour les étiquettes. Si non précisé, le design par défaut sera utilisé."),
    build_pngs: bool = typer.Option(False, "--png", help="Générer des PNGs à la place du PDF"),

    encoding: str = typer.Option("utf8", "--encoding", help="Encodage du fichier JSON"),
    # test_design: Optional[str] = typer.Option(None, "--test-design", help="Design à tester explicitement. test design X. Ce design sera le seul à être lu dans les plugins et son fichier de configuration sera considéré être celui de la société."),

    output: Optional[Path] = typer.Option(None, "--output", "-o", help="Fichier PDF ou PNG de sortie"),
    max: int = typer.Option(0, "--max", help="Nombre maximum d’étiquettes à générer"),
    debug: int = typer.Option(0, "--debug", help="Niveau de debug (0 à 3)"),
    paper: Optional[str] = typer.Option(None, "--paper", "-p", help="Taille du papier d'impression (ex: A3, A4)"),
    verbose: bool = typer.Option(False, "--verbose", "-v", help="Affiche les logs LaTeX"),
    no_latex: bool = typer.Option(False, "--no-latex", "--no-compile", help="Ne compile pas le fichier LaTeX (génère uniquement les fichiers sources)"),
    tarname: Optional[str] = typer.Option(None, "--tarname", help="Nom du fichier tar à générer (optionnel) dans le home"),
    attached_pid: int = typer.Option(0, "--attach-pid", help="Attache le PID du processus à la commande pour échange de données entre processus"),
):
    """
    Génère un fichier PDF à partir d’un fichier JSON d’articles.
    """
    _result = generate_labels_from_json(
        json_file=json_file,
        config_file=config_file,
        design_shortname=design,
        output=output,
        paper_size=paper,
        max_labels=max,
        encoding=encoding,
        debug=debug,
        no_latex=no_latex,
        verbose=verbose,
        # test_design=test_design,
        build_pngs=build_pngs,
        tarname=tarname,
        attached_pid=attached_pid
    )
    typer.Exit(0)

@app.command("generate-previews")
def generate_previews_command(
    force: bool = typer.Option(False, "--force", "-f", help="Force la régénération même si les aperçus existent"),
    design: Optional[str] = typer.Option(None, "--design", "-d", help="Ne traiter que ce design spécifique"),
    verbose: bool = typer.Option(False, "--verbose", "-v", help="Affichage détaillé"),
    output_dir: Optional[Path] = typer.Option(None, "--output-dir", "-o", help="Répertoire de sortie (par défaut: /home/local/etc/gcv6/etiquettes/design)"),
    no_compile: bool = typer.Option(False, "--no-compile", "--no-latex", help="Ne compile pas les fichiers LaTeX (génère uniquement les fichiers sources)"),
    debug: bool = typer.Option(False, "--debug", help="active le mode debug pour afficher des informations détaillées lors de la génération des aperçus"),
    enabled_shared_memory: bool = typer.Option(False, "--shared-memory", help="Active la communication par mémoire partagée pour suivre la progression")
):
    """
    Génère des aperçus PNG pour tous les formats d'étiquettes disponibles.
    """
    import logging

    # Configuration du logging
    log_level = logging.DEBUG if verbose else logging.INFO
    logging.basicConfig(level=log_level, format='%(levelname)s: %(message)s')

    sanity_check_succeed = True
    # Vérifier que lualatex est disponible
    if not check_lualatex_installed():
        typer.echo("Erreur: lualatex n'est pas installé", err=True)
        typer.echo("Veuillez installer lualatex: sudo apt install texlive-luatex ou équivalent", err=True)
        sanity_check_succeed = False

    # Vérifier que arara est disponible
    if not check_arara_installed():
        typer.echo("Erreur: arara n'est pas installé", err=True)
        typer.echo("Veuillez installer arara avec texlive", err=True)
        sanity_check_succeed = False

    if not check_pdftocairo_installed():
        typer.echo("Erreur: pdftocairo n'est pas installé", err=True)
        typer.echo("Veuillez installer poppler-utils: yum install poppler-utils ou dnf install poppler-utils", err=True)
        sanity_check_succeed = False

    if not sanity_check_succeed:
        raise typer.Exit(1)

    # Modifier le répertoire de sortie si spécifié
    if output_dir:
        from .generate_previews import PREVIEW_OUTPUT_DIR
        PREVIEW_OUTPUT_DIR = output_dir

    typer.echo(f"Génération des aperçus...")

    # Générer les aperçus avec attached_pid si mémoire partagée activée
    attached_pid = 0
    if enabled_shared_memory:
        import os
        attached_pid = os.getpid()
        typer.echo(f"Mémoire partagée activée (PID: {attached_pid})")
        typer.echo(f"   Surveillez avec: python test_shared_memory.py monitor {attached_pid}")


    results = generate_all_previews(
                    force=force, design_filter=design, output_dir=output_dir,
                    no_compile=no_compile, verbose=verbose, debug=debug,
                    attached_pid=attached_pid
            )

    # Afficher les résultats
    total_generated = 0
    total_failed = 0

    if debug:
        # Sauvegarde du dictionnaire results pour debugging
        import pickle
        from datetime import datetime

        debug_file = Path.home() / "gescokit_debug.pkl"
        debug_data = {
            "timestamp": datetime.now().isoformat(),
            "command": "generate-previews",
            "parameters": {
                "force": force,
                "design_filter": design,
                "output_dir": str(output_dir) if output_dir else None,
                "no_compile": no_compile
            },
            "results": results
        }

        with open(debug_file, "wb") as f:
            pickle.dump(debug_data, f)

        typer.echo(f"Données de debug sauvegardées dans: {debug_file}")
    typer.echo(f"\nRésultats de la génération des aperçus :")
    for design_name, design_info in results.items():
        design_path = design_info["design_path"]

        typer.echo(f"\nDesign: {design_name} → {design_path}")

        for format_name, format_results in design_info.get("formats", {}).items():
            categories = format_results["categories"]

            for category, result in categories.items():
                success = result["success"]
                path = result["path"]
                status = "OK" if success else "KO"
                if success and path:
                    typer.echo(f"  {format_name} - {category} : {status} → {path}")
                else:
                    typer.echo(f"  {format_name} - {category} : {status}")
                if success:
                    total_generated += 1
                else:
                    total_failed += 1

    typer.echo(f"\nRésumé: {total_generated} aperçus générés, {total_failed} échecs")

    if total_failed > 0:
        raise typer.Exit(1)

@app.command("generate-examples")
def generate_examples_command(
    design: Optional[str] = typer.Option(None, "--design", "-d", help="Design spécifique à utiliser"),
    format: Optional[str] = typer.Option(None, "--format", "-f", help="Format spécifique à utiliser"),
    output: Optional[Path] = typer.Option(None, "--output", "-o", help="Fichier PDF de sortie"),
    paper: str = typer.Option("a4", "--paper", "-p", help="Taille du papier (a4, a3, etc.)"),
    verbose: bool = typer.Option(False, "--verbose", "-v", help="Affichage détaillé"),
    no_compile: bool = typer.Option(False, "--no-compile", "--no-latex", help="Ne compile pas le fichier LaTeX (génère uniquement les fichiers sources)")
):
    """
    Génère un PDF d'exemples montrant différentes topologies d'un format d'étiquette.
    """
    import logging
    import subprocess

    # Configuration du logging
    log_level = logging.DEBUG if verbose else logging.INFO
    logging.basicConfig(level=log_level, format='%(levelname)s: %(message)s')

    # Vérifier que lualatex est disponible si compilation nécessaire
    if not no_compile:
        try:
            subprocess.run(["lualatex", "--version"], capture_output=True, check=True)
        except (subprocess.CalledProcessError, FileNotFoundError):
            typer.echo("Erreur: lualatex n'est pas disponible", err=True)
            typer.echo("Veuillez installer lualatex: sudo apt install texlive-luatex ou équivalent", err=True)
            raise typer.Exit(1)

    # Générer les exemples
    result = generate_examples(
        design=design,
        format=format,
        output=output,
        paper=paper,
        verbose=verbose,
        no_compile=no_compile
    )

    if result:
        typer.echo(f"Fichier d'exemples généré: {result}")
    else:
        typer.echo("Aucun exemple généré.")
        raise typer.Exit(1)

