# gescokit/crm/adel_crmya/__main__.py

import json

import typer

from gescokit.crm.adelya import Adelya
from gescokit.pyisql import crm as pyisql_crm

app = typer.Typer(help="Outils CRM Adelya")

client_app = typer.Typer(help="Commandes liées aux membres fidélité")


def get_adelya_instance() -> Adelya:
    """
    Initialise une instance Adelya avec configuration depuis la base de données.

    Returns
    -------
    Adelya
        Instance initialisée et prête à l'emploi.

    Raises
    ------
    ValueError
        Si aucun CRM actif n'est trouvé dans la base de données.
    """
    # Récupérer la configuration CRM depuis la base de données
    try:
        crm_config = pyisql_crm.get_active_crm()
    except ValueError as e:
        raise ValueError(f"Impossible de récupérer la configuration CRM : {e}")

    return Adelya(crm_config, crm_config["country"])


@client_app.command("by-cardnumber")
def fidMember_by_cardnumber(
    cardnumber: str = typer.Argument(..., help="Numéro de carte du client")
):
    """
    Recherche un membre fidélité à partir d'un numéro de carte.
    """
    adel_crm = get_adelya_instance()
    found, result = adel_crm.get_fidMember_by_cardnumber(cardnumber)
    if found:
        typer.echo(json.dumps(result))
    else:
        typer.echo("Aucun membre fidélité trouvé avec ce numéro de carte.")

@client_app.command("by-code")
def fidMember_by_code(
    code: str = typer.Argument(..., help="Code du client (code interne de la Gestion Commerciale)")
):
    """
    Récupère un membre fidélité à partir de son code interne dans la Gestion Commerciale.
    """
    adel_crm = get_adelya_instance()
    found, result = adel_crm.get_fidMember_by_id(code)
    if found:
        typer.echo(json.dumps(result))
    else:
        typer.echo("Aucun membre fidélité trouvé avec ce numéro de téléphone mobile.")


@client_app.command("by-mobile")
def fidMember_by_mobile(
    mobile: str = typer.Argument(..., help="Numéro de téléphone mobile du client")
):
    """
    Récupère un membre fidélité à partir d'un numéro de téléphone mobile.
    """
    adel_crm = get_adelya_instance()
    found, result = adel_crm.get_fidMember_by_mobile(mobile)
    if found:
        typer.echo(json.dumps(result))
    else:
        typer.echo("Aucun membre fidélité trouvé avec ce numéro de téléphone mobile.")

@client_app.command("added")
def fidMembers_added(
    from_date: str = typer.Argument(..., help="Date de début au format YYYY-MM-DD"),
    to_date: Optional[str] = typer.Argument(None, help="Date de fin au format YYYY-MM-DD")
):
    """
    Récupère tous les membres fidélité ajoutés à Adelya à partir d'une date (include).
    """
    adel_crm = get_adelya_instance()
    result = adel_crm.get_new_fidMembers(from_date, to_date)
    typer.echo(json.dumps(result))

@client_app.command("updated")
def fidMembers_updated(
    from_date: str = typer.Argument(..., help="Date de début au format YYYY-MM-DD"),
    to_date: Optional[str] = typer.Argument(None, help="Date de fin au format YYYY-MM-DD")
):
    """
    Récupère tous les membres fidélité modifiés (inclus la création) à partir d'une date (include).
    """
    adel_crm = get_adelya_instance()
    result = adel_crm.get_updated_fidMembers(from_date, to_date)
    typer.echo(json.dumps(result))


# ======= ======= ======= ======= =======  Nouvelle commande pour les AdEvents ======= ======= ======= ======= ======= 

adevent_app = typer.Typer(help="Commandes liées aux événements publicitaires")

@adevent_app.command("by-numfac")
def adEvent_by_numfac(
    numfac: str = typer.Argument(..., help="Code du client (code interne de la Gestion Commerciale)"),
    shopId: Optional[str] = typer.Option(None, "--shopId", "-s", help="Identifiant du magasin"),
    with_details: bool = typer.Option(False, "--with-details", "-d", help="Inclure les détails de la transaction")
):
    """
    Récupère une facture à partir de son code interne dans la Gestion Commerciale.
    """
    adel_crm = get_adelya_instance()
    result = adel_crm.get_adEvent_by_numfac(numfac, shopId=shopId, with_details=with_details)

    typer.echo(json.dumps(result))

@adevent_app.command("by-cardnumber")
def adEvent_by_cardnumber(
    cardnumber: str = typer.Argument(..., help="Numéro de carte du client")
):
    """
    Récupère les événements de liés à la ventes du client avec le numéro de fidélité.
    """
    adel_crm = get_adelya_instance()
    result = adel_crm.get_adEvent_by_cardnumber(cardnumber)

    typer.echo(json.dumps(result))

app.add_typer(client_app, name="client")
app.add_typer(adevent_app, name="adevent")
