from pathlib import Path
from typing import Callable, Optional, Union, Any
import rich
from rich.table import Table
from rich.console import Console

class FixedWidthParser:
    """
    Parser de ligne à largeur fixe, basé sur une liste de champs (nom, largeur [, transform]).
    """
    fields: list[tuple[str, Union[int, tuple[int, Callable[[str], object]]]]]
    metadata: dict[str, dict[str, Any]] = {}
    version: str = "1.0.0"
    version_gifi: str = "1.0.0"
    label: str = "Fichier à largeur fixe"

    def parse_line(self, line: str) -> dict:
        idxchar = 0 # Index du premier caractère du mot suivant
        result = {}
        for field in self.fields:
            name, width = field
            if isinstance(width, tuple):
                w, transform = width
            else:
                w, transform = width, str.strip
            result[name] = transform(line[idxchar:idxchar+w])
            idxchar += w
        return result

    def parse_file(self, path: Path) -> list[dict]:
        with path.open(encoding="utf-8") as f:
            return [self.parse_line(line.rstrip()) for line in f if line.strip()]


# def doc_fields(parser: FixedWidthParser) -> str:
#     lines = ["Nom du champ       Largeur"]
#     lines.append("------------------ -------")
#     for name, width in parser.fields:
#         w = width if isinstance(width, int) else width[0]
#         lines.append(f"{name:18} {w:>7}")
#     return "\n".join(lines)

def doc_fields(parser: FixedWidthParser) -> None:
    """
    Affiche une documentation enrichie des champs d'un parser FixedWidth.
    """
    console = Console()
    console.print(f"[bold underline]Documentation - {parser.__class__.__name__}[/]  (v{parser.version})")
    console.print(parser.label, style="dim")

    table = Table(show_header=True, header_style="bold magenta")
    table.add_column("Nom")
    table.add_column("Longueur", justify="right")
    table.add_column("Nom complet")
    table.add_column("Obligatoire")
    table.add_column("Description")

    for field in parser.fields:
        name, width = field
        w = width if isinstance(width, int) else width[0]
        meta = parser.metadata.get(name, {})
        table.add_row(
            name,
            str(w),
            meta.get("fullname", ""),
            "Oui" if meta.get("obligatoire") else "Non",
            meta.get("description", "")
        )

    console.print(table)


# CLI factory
def create_parser_cli(name: str, parser_cls: type[FixedWidthParser]):
    import typer
    import pandas as pd

    app = typer.Typer(help=f"Commandes liées au fichier {name.upper()}")

    @app.command("to-csv")
    def to_csv(input: Path, output: Optional[Path] = None):
        """
        Convertit un fichier à largeur fixe en CSV.

        Parameters
        ----------
        input : Path
            Chemin vers le fichier d'entrée.
        output : Path, optional
            Chemin de sortie CSV. Si omis, ajoute .csv au nom d'origine.
        """
        parser = parser_cls()
        data = parser.parse_file(input)
        output = output or input.with_suffix(".csv")
        pd.DataFrame(data).to_csv(output, index=False)
        typer.echo(f"Fichier exporté : {output}")

    @app.command("doc")
    def show_doc():
        """
        Affiche la documentation enrichie du format.
        """
        parser = parser_cls()
        doc_fields(parser)

    return app
