#!/usr/bin/env python3
"""
Script de lancement du middleware GCV
Usage: python run.py --config=config.toml [OPTIONS]
"""
import sys
import os
from pathlib import Path

import typer
from rich.console import Console
from rich.panel import Panel
from rich.text import Text

# Ajouter le dossier app au path pour les imports
sys.path.insert(0, os.path.join(os.path.dirname(__file__), 'app'))

app = typer.Typer(
    help="Middleware GCV - Routeur d'événements vers APIs externes",
    add_completion=False
)
console = Console()


def validate_config_file(config_path: Path) -> Path:
    """
    Valide que le fichier de configuration existe et est lisible

    Args:
        config_path: Chemin vers le fichier de configuration

    Returns:
        Path object du fichier validé

    Raises:
        typer.Exit: Si le fichier n'existe pas ou n'est pas valide
    """
    # Vérifier que le fichier existe
    if not config_path.exists():
        console.print(f"[red]ERREUR:[/red] Le fichier de configuration n'existe pas: {config_path}")
        console.print("\n[yellow]Créez le fichier à partir de l'exemple:[/yellow]")
        console.print(f"  [cyan]cp config.toml.example {config_path}[/cyan]")
        raise typer.Exit(1)

    # Vérifier que c'est bien un fichier
    if not config_path.is_file():
        console.print(f"[red]ERREUR:[/red] {config_path} n'est pas un fichier")
        raise typer.Exit(1)

    # Vérifier que c'est un fichier .toml
    if config_path.suffix != '.toml':
        console.print(f"[yellow]ATTENTION:[/yellow] Le fichier ne semble pas être un fichier TOML (.toml)")

    # Vérifier que le fichier est lisible
    if not os.access(config_path, os.R_OK):
        console.print(f"[red]ERREUR:[/red] Impossible de lire le fichier: {config_path}")
        raise typer.Exit(1)

    return config_path.resolve()


@app.command()
def main(
    config: Path = typer.Option(
        ...,
        "--config", "-c",
        help="Chemin vers le fichier de configuration config.toml (obligatoire)",
        exists=False,  # On gère la validation nous-mêmes
        dir_okay=False,
        resolve_path=True,
    ),
    dev: bool = typer.Option(
        False,
        "--dev",
        help="Démarrer en mode développement avec auto-reload",
    ),
    port: int = typer.Option(
        8001,
        "--port", "-p",
        help="Port d'écoute",
    ),
    host: str = typer.Option(
        "0.0.0.0",
        "--host",
        help="Adresse d'écoute",
    ),
    # workers: int = typer.Option(
    #     1,
    #     "--workers", "-w",
    #     min=1,
    #     max=16,
    #     help="Nombre de workers en mode production",
    # ),
):
    """
    Lance le middleware GCV avec la configuration spécifiée.

    Exemples:

        # Mode développement
        python run.py --config=config.toml --dev

        # Mode production
        python run.py --config=config.toml

        # Avec port personnalisé
        python run.py --config=config.toml --port=8080

        # Avec chemin absolu
        python run.py --config=/etc/gcv/middleware/config.toml
    """
    import uvicorn

    # Valider le fichier de configuration
    config_path = validate_config_file(config)

    # Définir les variables d'environnement
    os.environ["MIDDLEWARE_CONFIG_PATH"] = str(config_path)
    os.environ["MIDDLEWARE_DEV_MODE"] = "true" if dev else "false"

    # Afficher les informations de démarrage
    if dev:
        mode_text = Text("MODE DÉVELOPPEMENT", style="bold yellow")
        log_level = "info"
    else:
        mode_text = Text("MODE PRODUCTION", style="bold green")
        log_level = "warning"

    info_lines = [
        f"Configuration: {config_path}",
        f"URL: http://{host}:{port}",
        f"API Docs: http://{host}:{port}/docs",
        f"API Docs: http://{host}:{port}/redoc",
    ]

    workers = 1
    if not dev:
        info_lines.append(f"Workers: {workers}")

    panel_content = "\n".join(info_lines)
    console.print(Panel(
        panel_content,
        title=mode_text,
        border_style="cyan" if dev else "green",
        padding=(1, 2)
    ))

    # Lancer uvicorn
    if dev:
        uvicorn.run(
            "app.main:app",
            host=host,
            port=port,
            reload=True,
            log_level=log_level
        )
    else:
        uvicorn.run(
            "app.main:app",
            host=host,
            port=port,
            workers=workers,
            log_level=log_level
        )


if __name__ == "__main__":
    app()
