#!/usr/bin/env python3
"""
Script principal pour exécuter toutes les migrations

Ce script exécute toutes les migrations dans l'ordre numérique.
Les migrations sont nommées selon le format: NNN_description.py

Usage:
    python migrations/run_migrations.py --config=/path/to/config.toml
    python migrations/run_migrations.py [dbpath]

    --config: Lit fne.db depuis le fichier config.toml
    dbpath: Chemin direct vers la base de données
    Si aucun argument, utilise ./data/middleware.db par défaut
"""
import sys
import os
from pathlib import Path
import importlib.util
import tomllib


def get_migration_files(migrations_dir: Path) -> list:
    """
    Récupère tous les fichiers de migration triés par numéro

    Args:
        migrations_dir: Dossier contenant les migrations

    Returns:
        Liste de tuples (numéro, nom_fichier, chemin_complet)
    """
    migrations = []

    for file_path in migrations_dir.glob("*.py"):
        filename = file_path.name

        # Ignorer les fichiers spéciaux
        if filename.startswith("__") or filename == "run_migrations.py":
            continue

        # Vérifier le format NNN_*.py
        if "_" in filename and filename[0].isdigit():
            try:
                number = int(filename.split("_")[0])
                migrations.append((number, filename, file_path))
            except ValueError:
                print(f"[ATTENTION] Fichier ignoré (format invalide): {filename}")

    # Trier par numéro
    migrations.sort(key=lambda x: x[0])
    return migrations


def load_and_run_migration(migration_path: Path, dbpath: str) -> bool:
    """
    Charge et exécute une migration

    Args:
        migration_path: Chemin vers le fichier de migration
        dbpath: Chemin vers la base de données

    Returns:
        True si la migration a réussi, False sinon
    """
    # Charger le module dynamiquement
    spec = importlib.util.spec_from_file_location("migration", migration_path)
    if spec is None or spec.loader is None:
        print(f"[ERREUR] Impossible de charger la migration: {migration_path}")
        return False

    module = importlib.util.module_from_spec(spec)

    try:
        spec.loader.exec_module(module)
    except Exception as e:
        print(f"[ERREUR] Échec du chargement du module: {e}")
        return False

    # Exécuter la migration
    if hasattr(module, 'migrate_database'):
        return module.migrate_database(dbpath)
    elif hasattr(module, 'add_request_payload_column'):
        return module.add_request_payload_column(dbpath)
    else:
        print(f"[ERREUR] Fonction de migration introuvable dans {migration_path.name}")
        return False


def get_dbpath_from_config(config_path: str) -> str:
    """
    Lit dbpath depuis le fichier config.toml

    Args:
        config_path: Chemin vers le fichier config.toml

    Returns:
        Chemin vers la base de données
    """
    config_file = Path(config_path)
    if not config_file.exists():
        print(f"[ERREUR] Fichier de configuration introuvable: {config_path}")
        sys.exit(1)

    with open(config_file, "rb") as f:
        config = tomllib.load(f)

    fne_config = config.get("fne", {})
    dbpath = fne_config.get("db")
    if dbpath is None:
        print("[ERREUR] 'fne.db' non défini dans le fichier de configuration")
        sys.exit(1)

    # Si le chemin est relatif, le rendre relatif au dossier du config
    if not os.path.isabs(dbpath):
        dbpath = str(config_file.parent / dbpath)

    return dbpath


def main():
    """Point d'entrée du script"""

    # Parser les arguments - valeur par défaut
    dbpath = "./data/middleware.db"

    if len(sys.argv) > 1:
        arg = sys.argv[1]

        # Option --config=path ou --config path
        if arg.startswith("--config="):
            config_path = arg.split("=", 1)[1]
            dbpath = get_dbpath_from_config(config_path)
        elif arg == "--config" and len(sys.argv) > 2:
            config_path = sys.argv[2]
            dbpath = get_dbpath_from_config(config_path)
        elif arg.startswith("-"):
            print(f"[ERREUR] Option inconnue: {arg}")
            print("Usage: python run_migrations.py --config=/path/to/config.toml")
            print("       python run_migrations.py [dbpath]")
            sys.exit(1)
        else:
            # Chemin direct vers la base
            dbpath = arg

    # Vérifier que dbpath est défini
    if not os.path.exists(dbpath):
        print(f"[ERREUR] Base de données introuvable: {dbpath}")
        sys.exit(1)

    # Dossier des migrations
    migrations_dir = Path(__file__).parent

    # Récupérer toutes les migrations
    migrations = get_migration_files(migrations_dir)

    if not migrations:
        print("[INFO] Aucune migration à exécuter")
        sys.exit(0)

    print("=" * 70)
    print("Exécution des migrations")
    print("=" * 70)
    print(f"Base de données: {dbpath}")
    print(f"Migrations trouvées: {len(migrations)}")
    print()

    # Exécuter chaque migration
    success_count = 0
    failed_count = 0

    for number, filename, file_path in migrations:
        print(f"[{number:03d}] {filename}")
        print("-" * 70)

        success = load_and_run_migration(file_path, dbpath)

        if success:
            success_count += 1
            print("[OK] Migration terminée")
        else:
            failed_count += 1
            print("[ERREUR] Migration échouée")

        print()

    # Résumé
    print("=" * 70)
    print("Résumé des migrations")
    print("=" * 70)
    print(f"Total: {len(migrations)}")
    print(f"Réussies: {success_count}")
    print(f"Échouées: {failed_count}")
    print()

    if failed_count == 0:
        print("[OK] Toutes les migrations ont été appliquées avec succès")
        sys.exit(0)
    else:
        print(f"[ERREUR] {failed_count} migration(s) ont échoué")
        sys.exit(1)


if __name__ == "__main__":
    main()
