# Migrations de la base de données

Ce dossier contient les scripts de migration pour la base de données du middleware.

## Convention de nommage

Les migrations suivent le format : `NNN_description.py`
- `NNN` : Numéro de séquence à 3 chiffres (001, 002, etc.)
- `description` : Description courte de la migration

## Liste des migrations

### 001 - Ajout de error_details (2025-10-15)
**Fichier:** `001_add_error_details_column.py`

Ajoute une colonne `error_details` (TEXT, nullable) à la table `queue` pour stocker les détails des erreurs de validation et de traitement en format JSON.

### 002 - Ajout de request_payload (2025-11-12)
**Fichier:** `002_add_request_payload_column.py`

Ajoute une colonne `request_payload` (TEXT, nullable) à la table `api_responses` pour stocker le payload formaté envoyé aux APIs externes.

## Application des migrations

### Méthode 1: Script principal (recommandé)

Exécute toutes les migrations dans l'ordre :

```bash
cd /path/to/middleware
python migrations/run_migrations.py
# Ou avec un chemin spécifique
python migrations/run_migrations.py /path/to/middleware.db
```

### Méthode 2: Migration individuelle

```bash
python migrations/001_add_error_details_column.py [dbpath]
python migrations/002_add_request_payload_column.py [dbpath]
```

## Format des données stockées

### error_details (table queue)

#### Erreur de validation
```json
{
  "type": "validation_error",
  "validator": "validate_gcvx_invoice",
  "errors": [
    "emetteur.societe_nom: field required",
    "articles: field required"
  ],
  "warnings": [],
  "timestamp": "2025-10-15T10:30:45.123456"
}
```

#### Erreur de traitement
```json
{
  "type": "processing_error",
  "error": "Connection timeout",
  "traceback": "Traceback (most recent call last):\n  ...",
  "timestamp": "2025-10-15T10:30:45.123456"
}
```

### request_payload (table api_responses)

Contient le payload JSON formaté envoyé à l'API externe :

```json
{
  "invoiceType": "sale",
  "paymentMethod": "cash",
  "template": "B2C",
  "items": [{
    "designation": "Produit test",
    "quantity": 1,
    "amount": 1000,
    "discount": 0,
    "taxes": ["A"]
  }]
}
```

## Récupération des données via l'API

### Obtenir les détails complets d'une requête

```bash
curl http://localhost:8001/request/{request_id} | jq .
```

Réponse incluant toutes les données :

```json
{
  "request_id": 123,
  "status": "completed",
  "original_data": {
    // Données GCVX originales
  },
  "error_details": null,
  "api_responses": [
    {
      "destination": "fne_cdi_dev",
      "request_payload": {
        // Données formatées envoyées à FNE
      },
      "status_code": 200,
      "response_data": {
        // Réponse de l'API FNE
      }
    }
  ]
}
```

### Lister toutes les requêtes avec erreurs

```bash
curl http://localhost:8001/queue/all?status=failed | jq .
```

## Compatibilité

- **Base de données:** SQLite 3.x
- **Rétrocompatibilité:** Oui - les enregistrements existants auront les nouveaux champs à `NULL`
- **Impact:** Aucun sur les fonctionnalités existantes
- **SQLite minimum:** 3.7.11 (pour ALTER TABLE ADD COLUMN)

## Création d'une nouvelle migration

1. Créer un nouveau fichier avec le prochain numéro de séquence :
   ```bash
   touch migrations/003_ma_nouvelle_migration.py
   chmod +x migrations/003_ma_nouvelle_migration.py
   ```

2. Utiliser le template suivant :

```python
#!/usr/bin/env python3
"""
Migration NNN: Description de la migration

Date: YYYY-MM-DD
"""
import sqlite3
import sys
import os


def apply_migration(dbpath: str) -> bool:
    """
    Applique la migration

    Args:
        dbpath: Chemin vers la base de données

    Returns:
        True si la migration a réussi, False sinon
    """
    if not os.path.exists(dbpath):
        print(f"[ERREUR] Base de données introuvable: {dbpath}")
        return False

    try:
        conn = sqlite3.connect(dbpath)
        cursor = conn.cursor()

        # Vérifier si la migration est déjà appliquée
        # ...

        # Appliquer la migration
        # cursor.execute("ALTER TABLE ...")
        # conn.commit()

        conn.close()
        print("[OK] Migration terminée avec succès")
        return True

    except sqlite3.Error as e:
        print(f"[ERREUR] SQLite: {e}")
        return False
    except Exception as e:
        print(f"[ERREUR] Inattendue: {e}")
        return False


# Alias pour compatibilité avec run_migrations.py
migrate_database = apply_migration


if __name__ == "__main__":
    dbpath = "./data/middleware.db"
    if len(sys.argv) > 1:
        dbpath = sys.argv[1]

    print(f"Migration: {dbpath}")
    success = apply_migration(dbpath)
    sys.exit(0 if success else 1)
```

3. Tester la migration individuellement avant de l'inclure dans run_migrations.py

## Rollback

SQLite ne supporte pas DROP COLUMN avant la version 3.35.

Pour les versions antérieures, le rollback nécessite de recréer la table sans la colonne.
