#!/usr/bin/env python3
"""
Migration pour ajouter la colonne request_payload à la table api_responses

Cette migration ajoute une colonne TEXT pour stocker le payload formaté
envoyé à l'API externe.

Usage:
    python migrations/add_request_payload_column.py [dbpath]

    Si dbpath n'est pas fourni, utilise ./data/middleware.db par défaut
"""
import sqlite3
import sys
import os
from pathlib import Path


def add_request_payload_column(dbpath: str) -> bool:
    """
    Ajoute la colonne request_payload à la table api_responses

    Args:
        dbpath: Chemin vers la base de données SQLite

    Returns:
        True si la migration a réussi, False sinon
    """
    if not os.path.exists(dbpath):
        print(f"Erreur: Base de données introuvable: {dbpath}")
        return False

    try:
        conn = sqlite3.connect(dbpath)
        cursor = conn.cursor()

        # Vérifier si la colonne existe déjà
        cursor.execute("PRAGMA table_info(api_responses)")
        columns = [row[1] for row in cursor.fetchall()]

        if 'request_payload' in columns:
            print("[OK] La colonne 'request_payload' existe déjà")
            conn.close()
            return True

        # Ajouter la colonne
        print("Ajout de la colonne 'request_payload' à la table 'api_responses'...")
        cursor.execute("""
            ALTER TABLE api_responses
            ADD COLUMN request_payload TEXT
        """)

        conn.commit()
        print("[OK] Colonne ajoutée avec succès")

        # Vérifier que la colonne a bien été ajoutée
        cursor.execute("PRAGMA table_info(api_responses)")
        columns = [row[1] for row in cursor.fetchall()]

        if 'request_payload' not in columns:
            print("[ERREUR] La colonne n'a pas été ajoutée correctement")
            conn.close()
            return False

        conn.close()
        print("[OK] Migration terminée avec succès")
        return True

    except sqlite3.Error as e:
        print(f"[ERREUR] SQLite: {e}")
        return False
    except Exception as e:
        print(f"[ERREUR] Inattendue: {e}")
        return False


def main():
    """Point d'entrée du script"""
    # Chemin par défaut
    dbpath = "./data/middleware.db"

    # Utiliser le chemin fourni en argument si présent
    if len(sys.argv) > 1:
        dbpath = sys.argv[1]

    print(f"Migration de la base de données: {dbpath}")
    print(f"Ajout de la colonne 'request_payload' à la table 'api_responses'")
    print()

    success = add_request_payload_column(dbpath)

    if success:
        print()
        print("Migration réussie !")
        print("La colonne 'request_payload' est maintenant disponible pour stocker")
        print("les données formatées envoyées aux APIs externes.")
        sys.exit(0)
    else:
        print()
        print("Échec de la migration")
        sys.exit(1)


if __name__ == "__main__":
    main()
