"""
Modèles Pydantic pour le middleware
"""
from pydantic import BaseModel, Field
from typing import Optional, Dict, Any, List
from datetime import datetime
from enum import Enum

class RequestStatus(str, Enum):
    PENDING = "pending"
    PROCESSING = "processing"
    COMPLETED = "completed"
    FAILED = "failed"
    RETRYING = "retrying"

class IncomingRequest(BaseModel):
    """Requête entrante dans le middleware"""
    event_type: str = Field(..., description="Type d'événement (order_created, inventory_update, etc.)")
    data: Dict[str, Any] = Field(..., description="Données JSON à traiter")
    priority: int = Field(default=5, ge=1, le=10, description="Priorité (1=haute, 10=basse)")
    metadata: Optional[Dict[str, Any]] = Field(default=None, description="Métadonnées optionnelles")

class QueuedRequest(BaseModel):
    """Requête dans la file d'attente"""
    id: Optional[int] = None
    event_type: str
    data: Dict[str, Any]
    priority: int
    metadata: Optional[Dict[str, Any]]
    status: RequestStatus = RequestStatus.PENDING
    retry_count: int = 0
    max_retries: int = 3
    created_at: datetime = Field(default_factory=datetime.utcnow)
    processed_at: Optional[datetime] = None

class APIRequest(BaseModel):
    """Requête vers une API externe"""
    queue_id: int
    destination: str
    url: str
    method: str = "POST"
    headers: Dict[str, str]
    payload: Dict[str, Any]
    timeout: int = 30

class APIResponse(BaseModel):
    """Réponse d'une API externe"""
    queue_id: int
    destination: str
    request_payload: Optional[Dict[str, Any]] = None  # Payload formaté envoyé à l'API
    status_code: Optional[int] = None
    response_data: Optional[Dict[str, Any]] = None
    error: Optional[str] = None
    duration_ms: Optional[float] = None
    timestamp: datetime = Field(default_factory=datetime.utcnow)

class RequestResult(BaseModel):
    """Résultat global d'une requête"""
    request_id: int
    event_type: str
    status: RequestStatus
    destinations: List[str]
    responses: List[APIResponse]
    created_at: datetime
    completed_at: Optional[datetime] = None