"""
Configuration centralisée de structlog pour le middleware
"""
import structlog
import logging
import sys
import os
from logging.handlers import RotatingFileHandler
from pathlib import Path

def setup_logging(level: str = "DEBUG", json_logs: bool = False, log_file: str = None):
    """
    Configure structlog pour l'application

    Args:
        level: Niveau de log (DEBUG, INFO, WARNING, ERROR)
        json_logs: Si True, utilise le format JSON, sinon format console lisible
        log_file: Chemin du fichier de log (None = stdout uniquement)
    """
    # Créer le dossier logs si nécessaire
    if log_file:
        log_dir = os.path.dirname(log_file)
        if log_dir:
            os.makedirs(log_dir, exist_ok=True)

    # Configuration du niveau de log
    log_level = getattr(logging, level.upper())

    # Créer le root logger
    root_logger = logging.getLogger()
    root_logger.setLevel(log_level)
    root_logger.handlers.clear()  # Nettoyer les handlers existants

    # Handler pour stdout
    console_handler = logging.StreamHandler(sys.stdout)
    console_handler.setLevel(log_level)
    console_handler.setFormatter(logging.Formatter("%(message)s"))
    root_logger.addHandler(console_handler)

    # Handler pour fichier avec rotation
    if log_file:
        file_handler = RotatingFileHandler(
            log_file,
            maxBytes=10 * 1024 * 1024,  # 10 MB
            backupCount=5,
            encoding='utf-8'
        )
        file_handler.setLevel(log_level)
        file_handler.setFormatter(logging.Formatter("%(message)s"))
        root_logger.addHandler(file_handler)

    # Processeurs communs
    processors = [
        structlog.contextvars.merge_contextvars,
        structlog.stdlib.add_logger_name,
        structlog.stdlib.add_log_level,
        structlog.stdlib.PositionalArgumentsFormatter(),
        structlog.processors.TimeStamper(fmt="iso"),
        structlog.processors.StackInfoRenderer(),
    ]

    if json_logs:
        # Format JSON pour production
        processors.append(structlog.processors.JSONRenderer())
    else:
        # Format console lisible pour développement
        processors.extend([
            structlog.processors.ExceptionRenderer(),
            structlog.dev.ConsoleRenderer(colors=True)
        ])

    structlog.configure(
        processors=processors,
        wrapper_class=structlog.stdlib.BoundLogger,
        context_class=dict,
        logger_factory=structlog.stdlib.LoggerFactory(),
        cache_logger_on_first_use=True,
    )


def get_logger(name: str = None) -> structlog.stdlib.BoundLogger:
    """
    Obtient un logger structlog

    Args:
        name: Nom du logger (généralement __name__ du module)

    Returns:
        Logger structlog configuré
    """
    return structlog.get_logger(name)


# Configuration par défaut au démarrage du module
# Le log_file peut être configuré via variable d'environnement
dbpath = os.getenv("DBPATH", None)
if dbpath:
    default_log_dir = Path("/home/log/") / Path(dbpath).stem 
    os.makedirs(default_log_dir, exist_ok=True)
    log_file = default_log_dir / "daily" / "middleware.log"
else:
    log_file = "./logs/middleware.log"
log_level = os.getenv("MIDDLEWARE_LOG_LEVEL", "DEBUG")

# En production, seul le format JSON est autorisé
# Le mode est déterminé par run.py via --dev
dev_mode = os.getenv("MIDDLEWARE_DEV_MODE", "false").lower() in ("true", "1", "yes")
json_logs = False if dev_mode else True
#json_logs = True

setup_logging(level=log_level, json_logs=json_logs, log_file=str(log_file))
