from typing import Optional, Literal, Union
from pydantic import BaseModel, Field
from app.pydantic_compat import make_field_validator, get_model_dict, PYDANTIC_V2


# Types de factures supportés (nettoyés des espaces)
FACTURE_TYPES = [
    "FACTURE",
    "AVOIR",
    "PROFORMA",
    "COMMANDE CLIENT",
    "LIVRAISON EN COMPTE",
    "BON DE RETOUR",
    "BON DE LIVRAISON",
    "TICKET DE CAISSE",
    "INTERVENTION",
    "DEVIS BTP",
    "DOSSIER DE VENTE",
    "RESERVATION",
    "DOSSIER D'AVOIR",
    "TICKET EN ATTENTE",
    "BON D'ENLEVEMENT",
    "BL (facture)",
    "BON RETOUR SUPPRIME",
    "PIECE CLOTUREE",
    "RESERV. UTILISEE",
    "RESERV. ANNULEE",
    "COMMANDE CLI LIVREE",
    "DOSSIER RESERVAT.",
    "ORDRE DE TRAVAIL",
    "ORDRE TRAVAIL FACT.",
    "ORDRE TRAVAIL ANNUL",
    "CMD. TYPE CLIENT",
    "BON D'ALLOCATION",
    "BON RETOUR MAGASIN",
    "FACTURE EN ATTENTE",
    "FACTURE ABONNEMENT",
    "RESERVATION/CONTRAT",
    "ENTRETIEN MACHINE",
    "RESERV. LOCATION",
    "COMMANDE CLI ANNUL.",
    "COMMANDE CLI FACT.",
    "PROFORMA PROSPECT",
    "FACTURE (ND)",
    "BON RET MAG SUPPR.",
    "FACTURE D'ACOMPTE",
    "INVOICE WEB",
    "SLIP WEB"
]

# Modes de paiement supportés (nettoyés des espaces)
PAYMENT_METHODS = [
    "ESPECES",
    "CHEQUE",
    "TRAITE",
    "VIREMENT",
    "CARTE/CB",
    "CHQ/TERM",
    "parAVOIR",
    "BON ACH.",
    "FIN MOIS",
    "MAXMOD",
    "DEVISE 1",
    "DEVISE 2",
    "DEVISE 3",
    "DEVISE 4",
    "DEVISE 5",
    "DEVISE 6",
    "DEVISE 7",
    "CLOT AV",
    "CRT CRED",
    "FACTOR",
    "BON FID.",
    "BON AVR",
    "CRT CADO",
    "BAFID."
]


class Vendeur(BaseModel):
    """Informations sur le vendeur"""
    code: str = Field(..., min_length=1, description="Code du vendeur")
    nom: str = Field(..., min_length=1, description="Nom du vendeur")
    numero_operateur: str = Field(..., min_length=1, description="Numéro d'opérateur")
    numero_caisse: int = Field(..., ge=0, description="Numéro de caisse")


class Emetteur(BaseModel):
    """Informations sur l'entreprise émettrice"""
    raison_sociale: Optional[str] = Field(None, min_length=0, description="Nom de la société")
    adresse: Optional[str] = Field(None, min_length=0, description="Adresse de la société")
    code_postal: Optional[str] = Field(None, min_length=0, description="Code postal")
    ville: Optional[str] = Field(None, min_length=0, description="Ville")
    pays: Optional[str] = Field(None, min_length=0, description="Pays")
    siret: Optional[str] = Field(None, min_length=0, description="Numéro SIRET")
    codnaf: Optional[str] = Field(None, description="Code NAF/APE")
    tvaintra: Optional[str] = Field(None, description="Numéro TVA intracommunautaire")
    rcs: Optional[str] = Field(None, description="Immatriculation RCS")
    form_juridique: Optional[str] = Field(None, description="Forme juridique")
    capital: Optional[str] = Field(None, description="Capital social")


class Adresse(BaseModel):
    """Adresse du client"""
    raison_sociale: str = Field(..., description="Raison sociale ou nom")
    ligne1: Optional[str] = Field(None, description="Ligne 1 de l'adresse")
    ligne2: Optional[str] = Field(None, description="Ligne 2 de l'adresse")
    ligne3: Optional[str] = Field(None, description="Ligne 3 de l'adresse")
    code_postal: Optional[str] = Field(None, description="Code postal")
    ville: Optional[str] = Field(None, description="Ville")
    pays: Optional[str] = Field(None, description="Pays")
    email: Optional[str] = Field(None, description="Adresse email")
    mobile: Optional[str] = Field(None, description="Numéro de téléphone")


class Client(BaseModel):
    """Informations du client"""
    type: Literal["particulier", "professionnel"] = Field(..., description="Type de client")
    code: str =  Field(..., min_length=6, description="Code interne du client")
    cardnumber: Optional[str] = Field(None, description="Numéro de carte fidélité")
    adresses: dict[str, Adresse] = Field(default_factory=dict, description="Adresses du client")


class TaxeDetail(BaseModel):
    """Détail d'une taxe appliquée à un article
    
    Example
    -------
        {
            "code":	2,
            "taux":	18,
            "nom":	"TVA",
            "montant":	42712.2
        }
    """
    code: int = Field(..., description="Code de la taxe")
    taux: float = Field(..., ge=0, description="Taux de la taxe en pourcentage")
    nom: str = Field(..., description="Nom de la taxe (ex: TVA, O.M., DAOM)")
    montant: float = Field(..., ge=0, description="Montant de la taxe")


class Article(BaseModel):
    """Article sur la facture"""
    numlig: int = Field(..., ge=1, description="Numéro de ligne")
    bar: Optional[str] = Field(None, description="Code-barres")
    code_article: str = Field(..., min_length=1, description="Code article")
    description: str = Field(..., min_length=1, description="Description de l'article")
    quantite: float = Field(..., gt=0, description="Quantité")
    montant: float = Field(..., description="Montant")
    pubase_ht: float = Field(..., description="Prix unitaire de base HT")
    composition: int = Field(..., description="Composition")
    famille: str = Field(..., description="Code famille")
    est_prestation: bool = Field(..., description="Indique si c'est une prestation")
    taxes: list[TaxeDetail] = Field(..., description="Liste des taxes appliquées")
    offre: Optional[str] = Field(None, description="Code offre")
    tx_remise: float = Field(..., ge=0, le=100, description="Taux de remise")
    est_promo: bool = Field(..., description="Indique si c'est une promotion")
    promo_dossier: Optional[str] = Field(None, description="Dossier de promotion")
    code_dossier: int = Field(..., description="Code dossier")
    valart: float = Field(..., description="Valeur article TTC")
    valartht: float = Field(..., description="Valeur article HT")
    baseom: Optional[float] = Field(None, description="Base OM")
    totbrut: float = Field(..., description="Total brut")
    totart: float = Field(..., description="Total article")
    totarttc: float = Field(..., description="Total article TTC")
    totbruttc: float = Field(..., description="Total brut TTC")
    totartbrut: float = Field(..., description="Total article brut")
    artrem: float = Field(..., description="Montant remise article")
    artremttc: float = Field(..., description="Montant remise article TTC")
    rempromo: float = Field(..., description="Remise promotion")
    artrempied: float = Field(..., description="Remise pied article")
    valral: float = Field(..., description="Valeur ralliée")
    ecotaxe: float = Field(..., description="Écotaxe")
    taxepf1: Optional[float] = Field(None, description="Taxe PF1")
    taxepf2: Optional[float] = Field(None, description="Taxe PF2")
    taxepfp: Optional[float] = Field(None, description="Taxe PFP")
    valfrais: Optional[float] = Field(None, description="Valeur frais")
    taxecotis: Optional[float] = Field(None, description="Taxe cotisation")
    ecombl: Optional[float] = Field(None, description="Éco-mobilier")
    totbase: Optional[float] = Field(None, description="Total base")
    totbasettc: Optional[float] = Field(None, description="Total base TTC")
    points: Optional[float] = Field(None, description="Points fidélité")
    totartst: Optional[float] = Field(None, description="Total article stock")
    totarttcst: Optional[float] = Field(None, description="Total article TTC stock")
    valartst: Optional[float] = Field(None, description="Valeur article stock")
    valarthtst: Optional[float] = Field(None, description="Valeur article HT stock")
    artmargbrst: Optional[float] = Field(None, description="Marge brute article stock")

    @property
    def amount(self):
        return self.pubase_ht

    @property
    def quantity(self):
        return self.quantite


class Reglement(BaseModel):
    """Détails d'un règlement"""
    mode: str = Field(..., min_length=1, description="Mode de paiement")
    montant: float = Field(..., gt=0, description="Montant du règlement")
    # devise: Optional[str] = Field("XOF", description="Devise du règlement")
    info: Optional[str] = Field(None, description="Information diverse sur le règlement")

    @make_field_validator('mode')
    def validate_payment_method(cls, v: str) -> str:
        """Valide que le mode de paiement est dans la liste autorisée"""
        if v not in PAYMENT_METHODS:
            raise ValueError(f"Mode de paiement '{v}' non reconnu. Valeurs autorisées: {', '.join(PAYMENT_METHODS)}")
        return v


class Pied(BaseModel):
    """Pied de facture avec totaux et métadonnées"""
    version: str = Field(..., description="Version du logiciel")
    editee: int = Field(..., ge=0, description="Nombre d'éditions")
    remise: float = Field(..., ge=0, description="Montant de la remise")
    total_ttc: float = Field(..., description="Total TTC")
    total_ht: float = Field(..., description="Total HT")
    commentaires: list[str] = Field(default_factory=list, description="Liste de commentaires")
    sign_restitution: str = Field(default="", description="Signature de restitution")


class GCVXDocumentData(BaseModel):
    """
    Schéma de validation pour une facture de l'ERP GCVX
    """
    fac_num: str = Field(
        ...,
        min_length=10,
        description="Numéro de la facture"
    )
    fac_date: str = Field(
        ...,
        min_length=15,
        max_length=15,
        description="Date de la facture au format YYYYMMDD HHmmss"
    )
    client_code: str = Field(
        ...,
        min_length=6,
        description="Code interne du client"
    )
    fac_type: str = Field(
        ...,
        description="Type de la pièce"
    )
    num_caisse: int = Field(
        ...,
        ge=0,
        description="Numéro de caisse"
    )
    piece_origine: Optional[str] = Field(
        None,
        min_length=10,
        description="Numéro de la pièce d'origine"
    )
    vendeur: Vendeur = Field(..., description="Informations du vendeur")
    emetteur: Emetteur = Field(..., description="Informations de l'émetteur")
    # main_paymentMethod: str = Field(..., min_length=1, description="Méthode de paiement principale")
    client: Client = Field(..., description="Informations du client")
    articles: list[Article] = Field(..., min_items=1, description="Liste des articles facturés")
    reglements: list[Reglement] = Field(default_factory=list, description="Liste des règlements")
    pied: Pied = Field(..., description="Pied de facture avec totaux")

    @make_field_validator('fac_type')
    def validate_facture_type(cls, v: str) -> str:
        """Valide que le type de facture est dans la liste autorisée"""
        if v not in FACTURE_TYPES:
            raise ValueError(f"Type de facture '{v}' non reconnu. Valeurs autorisées: {', '.join(FACTURE_TYPES[:10])}...")
        return v

    # @make_field_validator('main_paymentMethod')
    # def validate_main_payment_method(cls, v: str) -> str:
    #     """Valide que la méthode de paiement principale est dans la liste autorisée"""
    #     if v not in PAYMENT_METHODS:
    #         raise ValueError(f"Méthode de paiement '{v}' non reconnue. Valeurs autorisées: {', '.join(PAYMENT_METHODS)}")
    #     return v

    def remove_articles_commentaires(self) -> dict:
        """Retourne un dict sans les articles et commentaires pour les logs"""
        data = get_model_dict(self)
        data['articles'] = "[SUPPRIMÉ]"
        data['pied']['commentaires'] = "[SUPPRIMÉS]"
        return data
    class Config:
        """Configuration Pydantic v1"""
        str_strip_whitespace = True
        json_schema_extra = {
            "example": {
                "fac_num": "FA25001205",
                "fac_date": "20250910205450",
                "client_code": "INFOCE",
                "fac_type": "FACTURE",
                "num_caisse": 1,
                "piece_origine": None,
                "vendeur": {
                    "code": "01",
                    "nom": "Jean Dupont",
                    "numero_operateur": "gcv6-op01",
                    "numero_caisse": 1
                },
                "emetteur": {
                    "societe_nom": "BRICOCERAM SARL",
                    "societe_adresse": "ZI LA JAMBETTE",
                    "societe_code_postal": "97232",
                    "societe_ville": "LAMENTIN",
                    "pays": "MARTINIQUE",
                    "siret": "51319447200025",
                    "codnaf": "4752B",
                    "tvaintra": "FR10513194472",
                    "rcs": "R.C.S. Fort-de-france",
                    "form_juridique": "SARL",
                    "capital": "2 358 800,00 Euros"
                },
                "main_paymentMethod": "ESPECES",
                "client": {
                    "type": "particulier",
                    "facturation": {
                        "raison_sociale": "Martin Kouassi",
                        "mobile": "+237677654321",
                        "email": "martin.kouassi@gmail.com"
                    }
                },
                "articles": [
                    {
                        "taxes": "TVAB",
                        "description": "Smartphone Samsung Galaxy A54",
                        "quantity": 1,
                        "amount": 185000
                    },
                    {
                        "taxes": "TVAB",
                        "description": "Coque de protection + film écran",
                        "quantity": 1,
                        "amount": 8500
                    }
                ],
                "reglements": [
                    {
                        "mode": "ESPECES",
                        "montant": 150000,
                        "devise": "XOF",
                        "reference": None
                    },
                    {
                        "mode": "CARTE/CB",
                        "montant": 43500,
                        "devise": "XOF",
                        "reference": "CB-20250910-1234"
                    }
                ],
                "pied": {
                    "version": "6.06.012ab",
                    "editee": 1,
                    "remise": 0,
                    "total_ttc": 193500,
                    "total_ht": 165000,
                    "commentaires": [
                        "Merci de votre visite",
                        "Garantie 12 mois sur le smartphone",
                        "Échange possible sous 14 jours"
                    ],
                    "sign_restitution": ""
                }
            }
        }
