"""
Validateur pour les factures au format GCVX
Utilise le schéma Pydantic GCVXDocumentData pour valider les données entrantes
"""
from typing import Dict, Any
from pydantic import ValidationError
from app.gcvx.validators.piece import GCVXDocumentData
from app.pydantic_compat import get_model_dict
from .base import ValidationResult


class GCVXInvoiceValidator:
    """
    Validateur pour les factures GCVX
    """

    @staticmethod
    def validate(data: Dict[str, Any]) -> ValidationResult:
        """
        Valide les données d'une facture GCVX

        Args:
            data: Données de la facture au format GCVX

        Returns:
            ValidationResult avec les erreurs éventuelles
        """
        errors = []

        try:
            # Validation via Pydantic
            validated_data = GCVXDocumentData(**data)

            # Validation métier supplémentaire
            business_errors = GCVXInvoiceValidator._validate_business_rules(validated_data)
            errors.extend(business_errors)

            return ValidationResult(
                valid=len(errors) == 0,
                errors=errors,
                data=get_model_dict(validated_data) if len(errors) == 0 else None
            )

        except ValidationError as e:
            # Convertir les erreurs Pydantic en format lisible
            for error in e.errors():
                field = ".".join(str(loc) for loc in error["loc"])
                message = error["msg"]
                errors.append(f"{field}: {message}")

            return ValidationResult(
                valid=False,
                errors=errors,
                data=None
            )
        except Exception as e:
            return ValidationResult(
                valid=False,
                errors=[f"Erreur inattendue lors de la validation: {str(e)}"],
                data=None
            )

    @staticmethod
    def _validate_business_rules(data: GCVXDocumentData) -> list[str]:
        """
        Valide les règles métier spécifiques

        Args:
            data: Données validées par Pydantic

        Returns:
            Liste des erreurs métier
        """
        errors = []

        # # Vérifier que les totaux sont cohérents
        # if data.pied.total_ht > data.pied.total_ttc:
        #     errors.append("pied.total_ht: Le total HT ne peut pas être supérieur au total TTC")

        # # Vérifier que la remise est logique
        # if data.pied.remise > data.pied.total_ttc:
        #     errors.append("pied.remise: La remise ne peut pas être supérieure au total TTC")

        # Vérifier qu'il y a au moins un article
        if not data.articles:
            errors.append("articles: Au moins un article est requis")

        # Vérifier que les montants des articles sont positifs
        for idx, article in enumerate(data.articles):
            if article.amount < 0:
                errors.append(f"articles[{idx}].amount: Le montant ne peut pas être négatif")
            if article.quantity <= 0:
                errors.append(f"articles[{idx}].quantity: La quantité doit être strictement positive")

        # Vérifier la cohérence du type de facture
        if data.fac_type == "AVOIR":
            # Pour un avoir, on peut avoir des montants négatifs
            if data.pied.total_ttc > 0:
                errors.append("pied.total_ttc: Un avoir devrait avoir un total négatif ou nul")

        return errors


def validate_gcvx_invoice(data: dict) -> ValidationResult:
    """
    Fonction de validation pour les factures GCVX

    Args:
        data: Données de la facture à valider

    Returns:
        ValidationResult avec les erreurs éventuelles
    """
    return GCVXInvoiceValidator.validate(data)
