"""
Validateurs pour les factures FNE-CDI (Côte d'Ivoire)
Conforme à l'API de signature électronique FNE
Compatible Pydantic v1 (CentOS 6) et v2 (versions modernes)
"""
from pydantic import BaseModel, Field
from typing import Optional, List, Literal
from .base import InvoiceItem, CustomTax, ValidationResult
from app.pydantic_compat import make_field_validator, make_model_validator, get_model_dict, PYDANTIC_V2
from ..config import FNE_CDI_POS, FNE_CDI_ESTABLISHMENT
from app.gcvx.validators import GCVXDocumentData

class FNEInvoiceData(BaseModel):
    """
    Schéma de validation pour une facture FNE-CDI
    Basé sur l'API: POST /external/invoices/sign
    """
    # Informations obligatoires
    invoiceType: Literal["sale", "purchase"] = Field(
        ...,
        description="Type de facture: sale (vente ou avoir), purchase ()"
    )
    paymentMethod: Literal["cash", "card", "check", "mobile-money", "transfer", "deferred"] = Field(
        ...,
        description="Méthode de paiement"
    )
    template: Literal["B2B", "B2C"] = Field(
        ...,
        description="Template de facture: B2B (entreprise) ou B2C (particulier)"
    )
    pointOfSale: str = Field(
        ...,
        min_length=1,
        max_length=100,
        description="Point de vente"
    )
    establishment: str = Field(
        ...,
        min_length=1,
        max_length=100,
        description="Établissement"
    )

    # Client - Obligatoire pour B2B, optionnel pour B2C
    clientNcc: Optional[str] = Field(
        None,
        max_length=50,
        description="Numéro de compte contribuable du client"
    )
    clientCompanyName: Optional[str] = Field(
        None,
        max_length=255,
        description="Raison sociale du client"
    )
    clientPhone: Optional[str] = Field(
        None,
        max_length=20,
        description="Téléphone du client"
    )
    clientEmail: Optional[str] = Field(
        None,
        max_length=255,
        description="Email du client"
    )
    clientSellerName: Optional[str] = Field(
        None,
        max_length=255,
        description="Nom du vendeur/commercial"
    )

    # Lignes de facture (obligatoire)
    items: List[InvoiceItem] = Field(
        ...,
        description="Lignes de la facture (au moins 1)"
    )

    # Informations optionnelles
    commercialMessage: Optional[str] = Field(
        None,
        max_length=500,
        description="Message commercial"
    )
    footer: Optional[str] = Field(
        None,
        max_length=500,
        description="Pied de page"
    )
    foreignCurrency: Optional[str] = Field(
        None,
        max_length=10,
        description="Code devise étrangère (EUR, USD, etc.)"
    )
    foreignCurrencyRate: Optional[float] = Field(
        None,
        ge=0,
        description="Taux de change"
    )
    customTaxes: List[CustomTax] = Field(
        default_factory=list,
        description="Taxes personnalisées au niveau facture"
    )
    discount: float = Field(
        default=0,
        ge=0,
        le=100,
        description="Remise globale en pourcentage"
    )

    @make_field_validator('items')
    @classmethod
    def validate_items_not_empty(cls, v):
        """Valide qu'il y a au moins une ligne de facture"""
        if not v or len(v) < 1:
            raise ValueError("La facture doit contenir au moins une ligne")
        return v

    @make_field_validator('clientPhone')
    @classmethod
    def validate_phone(cls, v):
        """Valide le format du téléphone"""
        if v is None:
            return v

        # Nettoyer le téléphone
        phone = v.replace(" ", "").replace("-", "").replace(".", "")
        if phone[0] == "+":
            phone = phone[1:]

        # Vérifier que ce sont bien des chiffres
        if not phone.isdigit():
            raise ValueError("Le numéro de téléphone ne doit contenir que des chiffres")

        # Vérifier la longueur (entre 8 et 15 chiffres)
        if len(phone) < 8 or len(phone) > 15:
            raise ValueError("Le numéro de téléphone doit contenir entre 8 et 15 chiffres")

        return phone

    @make_field_validator('clientEmail')
    @classmethod
    def validate_email(cls, v):
        """Valide le format de l'email"""
        if v is None:
            return v

        if '@' not in v or '.' not in v:
            raise ValueError("Format d'email invalide")

        return v.lower().strip()

    @make_field_validator('discount')
    @classmethod
    def validate_discount(cls, v):
        """Valide la remise"""
        if v < 0 or v > 100:
            raise ValueError("La remise doit être entre 0 et 100")
        return round(v, 2)

    # ===== Validateurs Pydantic v2 (signature moderne avec self) =====
    if PYDANTIC_V2:
        @make_model_validator(mode='after')
        def validate_client_info(self):
            """Valide les informations client selon le template"""
            if self.template == "B2B":
                # Pour B2B, certaines informations sont obligatoires
                if not self.clientNcc:
                    raise ValueError("clientNcc est obligatoire pour le template B2B")
                if not self.clientCompanyName:
                    raise ValueError("clientCompanyName est obligatoire pour le template B2B")

            return self

        @make_model_validator(mode='after')
        def validate_foreign_currency(self):
            """Valide la cohérence devise étrangère / taux"""
            if self.foreignCurrency and not self.foreignCurrencyRate:
                raise ValueError("foreignCurrencyRate est requis si foreignCurrency est défini")

            if not self.foreignCurrency and self.foreignCurrencyRate:
                raise ValueError("foreignCurrency est requis si foreignCurrencyRate est défini")

            return self

    # ===== Validateurs Pydantic v1 (signature legacy avec cls, values) =====
    else:
        @make_model_validator(mode='after')
        def validate_client_info(cls, values):
            """Valide les informations client selon le template (Pydantic v1)"""
            if values.get('template') == "B2B":
                # Pour B2B, certaines informations sont obligatoires
                if not values.get('clientNcc'):
                    raise ValueError("clientNcc est obligatoire pour le template B2B")
                if not values.get('clientCompanyName'):
                    raise ValueError("clientCompanyName est obligatoire pour le template B2B")

            return values

        @make_model_validator(mode='after')
        def validate_foreign_currency(cls, values):
            """Valide la cohérence devise étrangère / taux (Pydantic v1)"""
            if values.get('foreignCurrency') and not values.get('foreignCurrencyRate'):
                raise ValueError("foreignCurrencyRate est requis si foreignCurrency est défini")

            if not values.get('foreignCurrency') and values.get('foreignCurrencyRate'):
                raise ValueError("foreignCurrency est requis si foreignCurrencyRate est défini")

            return values


class FNECancelInvoiceData(BaseModel):
    """Schéma pour l'annulation d'une facture FNE"""
    invoiceId: str = Field(
        ...,
        min_length=1,
        description="ID de la facture à annuler"
    )
    reason: str = Field(
        ...,
        min_length=10,
        max_length=500,
        description="Raison de l'annulation"
    )
    establishment: str = Field(
        ...,
        min_length=1,
        description="Établissement"
    )


class FNEInvoiceValidator:
    """Validateur pour les factures FNE-CDI"""

    @staticmethod
    def validate(data: dict) -> ValidationResult:
        """
        Valide les données d'une facture FNE

        Args:
            data: Dictionnaire des données à valider

        Returns:
            ValidationResult avec les erreurs ou les données validées
        """
        result = ValidationResult(valid=True)

        try:
            # Ajouter les valeurs de configuration si absentes
            data_with_config = data | {
                "pointOfSale": data.get("pointOfSale", FNE_CDI_POS),
                "establishment": data.get("establishment", FNE_CDI_ESTABLISHMENT)
            }

            # Valider avec Pydantic
            validated_data = FNEInvoiceData(**data_with_config)

            # Validations métier supplémentaires
            result = FNEInvoiceValidator._business_validations(validated_data, result)

            if result.valid:
                # Convertir en dict pour le retour
                result.data = get_model_dict(validated_data)

        except Exception as e:
            result.add_error(f"Erreur de validation: {str(e)}")

        return result


    @staticmethod
    def _business_validations(invoice: FNEInvoiceData, result: ValidationResult) -> ValidationResult:
        """
        Validations métier supplémentaires

        Args:
            invoice: Données de la facture validées par Pydantic
            result: Résultat de validation à enrichir

        Returns:
            ValidationResult mis à jour
        """
        # Vérifier le montant total
        total_amount = sum(
            item.quantity * item.amount * (1 - item.discount / 100)
            for item in invoice.items
        )

        if total_amount <= 0:
            result.add_error("Le montant total de la facture doit être supérieur à 0")

        # Vérifier qu'il y a au moins une ligne
        if len(invoice.items) == 0:
            result.add_error("La facture doit contenir au moins une ligne")

        # Avertissement si pas de client pour B2C
        if invoice.template == "B2C" and not invoice.clientPhone and not invoice.clientEmail:
            result.add_warning("Aucune information de contact client pour une facture B2C")

        # Vérifier les taxes sur les lignes
        for idx, item in enumerate(invoice.items):
            if not item.taxes and not item.customTaxes:
                result.add_warning(f"Ligne {idx + 1}: Aucune taxe définie")

        return result

    @staticmethod
    def validate_cancel(data: dict) -> ValidationResult:
        """Valide une demande d'annulation de facture"""
        result = ValidationResult(valid=True)

        try:
            # Ajouter establishment depuis la config si absent
            data_with_config = data | {
                "establishment": data.get("establishment", FNE_CDI_ESTABLISHMENT)
            }

            validated_data = FNECancelInvoiceData(**data_with_config)
            result.data = get_model_dict(validated_data)
        except Exception as e:
            result.add_error(f"Erreur de validation: {str(e)}")

        return result
