"""
Schémas Pydantic de base pour la validation
Compatible Pydantic v1 (CentOS 6) et v2 (versions modernes)
"""
from pydantic import BaseModel, Field
from typing import Optional, List, Dict, Any
from decimal import Decimal
from app.pydantic_compat import make_field_validator, PYDANTIC_V2


class CustomTax(BaseModel):
    """Taxe personnalisée"""
    name: str = Field(..., min_length=1, max_length=100, description="Nom de la taxe")
    amount: float = Field(..., gt=0, description="Montant de la taxe")

    @make_field_validator('amount')
    @classmethod
    def validate_amount(cls, v):
        if v <= 0:
            raise ValueError("Le montant doit être supérieur à 0")
        return round(v, 2)


class InvoiceItem(BaseModel):
    """Ligne de facture"""
    reference: str = Field(..., min_length=1, max_length=255, description="Référence article")
    description: str = Field(..., min_length=1, max_length=500, description="Description")
    quantity: float = Field(..., gt=0, description="Quantité")
    amount: float = Field(..., gt=0, description="Montant unitaire")
    discount: float = Field(default=0, ge=0, le=100, description="Remise en pourcentage")
    measurementUnit: str = Field(default="pcs", max_length=50, description="Unité de mesure")
    taxes: List[str] = Field(default_factory=list, description="Liste des taxes (TVA, TVAC, etc.)")
    customTaxes: List[CustomTax] = Field(default_factory=list, description="Taxes personnalisées")

    @make_field_validator('quantity', 'amount')
    @classmethod
    def validate_positive(cls, v):
        if v <= 0:
            raise ValueError("La valeur doit être supérieure à 0")
        return round(v, 2)

    @make_field_validator('discount')
    @classmethod
    def validate_discount(cls, v):
        if v < 0 or v > 100:
            raise ValueError("La remise doit être entre 0 et 100")
        return round(v, 2)

    @make_field_validator('taxes')
    @classmethod
    def validate_taxes(cls, v):
        valid_taxes = {"TVA", "TVAC", "TPS", "EXONERE"}
        for tax in v:
            if tax not in valid_taxes:
                raise ValueError(f"Taxe invalide: {tax}. Valeurs acceptées: {valid_taxes}")
        return v


class ValidationResult(BaseModel):
    """Résultat de validation"""
    valid: bool = Field(..., description="True si les données sont valides")
    errors: List[str] = Field(default_factory=list, description="Liste des erreurs de validation")
    warnings: List[str] = Field(default_factory=list, description="Liste des avertissements")
    data: Optional[Dict[str, Any]] = Field(default=None, description="Données validées et normalisées")

    def add_error(self, message: str):
        """Ajoute une erreur"""
        self.valid = False
        self.errors.append(message)

    def add_warning(self, message: str):
        """Ajoute un avertissement"""
        self.warnings.append(message)

    if PYDANTIC_V2:
        # Pydantic v2: utilise model_config
        pass
    else:
        # Pydantic v1: utilise Config
        class Config:
            arbitrary_types_allowed = True
