"""
Mapper de transformation des données GCVX vers format FNE-CDI
Convertit les factures du format interne GCVX vers le format attendu par l'API FNE
"""
from typing import Dict, Any, List, Optional
from app.gcvx.validators.piece import GCVXDocumentData


# Mapping des types de documents
INVOICE_TYPE_MAPPING = {
    "FACTURE": "sale",
    "AVOIR": "sale",
}

# Mapping des modes de paiement
PAYMENT_METHOD_MAPPING = {
    "ESPECES": "cash",
    "CHEQUE": "check",
    "TRAITE": "deferred",
    "VIREMENT": "transfer",
    "CARTE/CB": "card",
    "mobile-money": "mobile-money"
}


class GCVXToFNEMapper:
    """
    Classe de transformation des données GCVX vers format FNE-CDI
    """

    @staticmethod
    def map_invoice(gcvx_data: GCVXDocumentData) -> Dict[str, Any]:
        """
        Transforme une facture GCVX en format FNE-CDI

        Args:
            gcvx_data: Données facture au format GCVX (modèle Pydantic validé)

        Returns:
            Données au format FNE-CDI
        """
        # Déterminer le template (B2B ou B2C)
        template = GCVXToFNEMapper._determine_template(gcvx_data)

        # Construire la structure FNE
        fne_data = {
            # Champs obligatoires
            "invoiceType": GCVXToFNEMapper._map_invoice_type(gcvx_data.fac_type),
            "paymentMethod": GCVXToFNEMapper._map_payment_method(gcvx_data.reglements[0].mode),
            "template": template,
            "clientSellerName": f"{gcvx_data.vendeur.nom} ({gcvx_data.vendeur.code})",

            # Items (lignes de facture)
            "items": GCVXToFNEMapper._map_items(gcvx_data.articles),

            # Remise globale
            "discount": gcvx_data.pied.remise
        }

        # Ajouter les informations client si présentes
        client_data = GCVXToFNEMapper._map_client(gcvx_data.client, template)
        fne_data.update(client_data)

        # Ajouter les informations optionnelles
        optional_data = GCVXToFNEMapper._map_optional_fields(gcvx_data)
        fne_data.update(optional_data)

        # Note: pointOfSale et establishment seront injectés par le validateur depuis la config

        return fne_data

    @staticmethod
    def _determine_template(gcvx_data: GCVXDocumentData) -> str:
        """
        Détermine si c'est une facture B2B ou B2C
        B2B si le client est de type "professionnel"

        Args:
            gcvx_data: Données GCVX

        Returns:
            "B2B" ou "B2C"
        """
        return "B2C" if gcvx_data.client.type == "particulier" else "B2B"

    @staticmethod
    def _map_invoice_type(fac_type: Optional[str]) -> str:
        """
        Convertit le type de facture GCV vers format FNE

        Args:
            fac_type: Type de facture GCV (FACTURE, AVOIR, DEVIS)

        Returns:
            Type FNE (sale, refund, proforma)
        """

        if not fac_type or fac_type.upper() not in INVOICE_TYPE_MAPPING:
            error_message = f"Type de facture non gérée par FNE CDI : {fac_type}"
            raise NotImplementedError(error_message)

        return INVOICE_TYPE_MAPPING[fac_type.upper()]

    @staticmethod
    def _map_payment_method(mode: str) -> str:
        """
        Convertit le mode de paiement GCVX vers format FNE

        Args:
            mode: Mode de paiement GCVX

        Returns:
            Mode de paiement FNE
        """
        # Le mode de paiement dans GCVX est déjà au format FNE
        return PAYMENT_METHOD_MAPPING.get(mode, "card")

    @staticmethod
    def _map_client(client: Any, template: str) -> Dict[str, Any]:
        """
        Mappe les informations client

        Args:
            client: Données client GCVX (modèle Client)
            template: Template (B2B ou B2C)

        Returns:
            Dictionnaire avec les champs client FNE
        """
        result = {}

        adresses = client.adresses
        facturation = adresses.get("facturation", {})
        livraison = adresses.get("livraison", {})

        # Raison sociale/nom (obligatoire pour tous)
        result["clientCompanyName"] = facturation.raison_sociale

        # Téléphone (normaliser en digits uniquement)
        if facturation.mobile:
            phone = facturation.mobile.replace(" ", "").replace("-", "").replace(".", "").strip()
            if phone:
                result["clientPhone"] = phone
        if not result.get("clientPhone", ""):
            # Fournir une valeur par défaut car ce champ est obligatoire
            result["clientPhone"] = "0000000000"

        # Email (normaliser en lowercase)
        if facturation.email:
            result["clientEmail"] = facturation.email.strip().lower()
        else:
            # Fournir une valeur par défaut car ce champ est obligatoire
            result["clientEmail"] = "no-email@valid.local"

        return result

    @staticmethod
    def _map_items(articles: List[Any]) -> List[Dict[str, Any]]:
        """
        Mappe les articles de facture

        Args:
            articles: Liste des articles GCVX

        Returns:
            Liste des items FNE
        """
        items = []

        for article in articles:
            item = {
                "reference": article.code_article,
                "description": article.description,
                "quantity": float(article.quantite),
                "amount": float(article.pubase_ht),
                "taxes": ["TVA"],
                "customTaxes": []  # GCVX ne supporte pas encore les taxes custom dans ce modèle
            }

            if (discount := article.tx_remise) != 0:
                item["discount"] = float(discount)

            for taxe in article.taxes:
                if taxe.nom == "TVA":
                    continue  # La TVA est gérée séparément
                if abs(taxe.montant) > 0.001:
                    tax_item = {
                        "name": taxe.nom,
                        "amount": float(taxe.taux)
                    }
                    item["customTaxes"].append(tax_item)

            if item["customTaxes"] == []:
                del item["customTaxes"]


            items.append(item)

        return items

    @staticmethod
    def _map_optional_fields(gcvx_data: GCVXDocumentData) -> Dict[str, Any]:
        """
        Mappe les champs optionnels

        Args:
            gcvx_data: Données GCVX complètes

        Returns:
            Dictionnaire avec les champs optionnels FNE
        """
        result = {}

        # result["commercialMessage"] = "...."
        # Pied de facture - commentaires comme message commercial
        result["footer"] = gcvx_data.fac_num
        if gcvx_data.pied.commentaires:
            # Joindre les commentaires non vides
            comments = [c for c in gcvx_data.pied.commentaires if c.strip()]
            if comments:
                result["footer"] +=  " " + " ".join(comments)
        
        # # Taxes personnalisées au niveau facture (vide pour l'instant)
        # result["customTaxes"] = []

        return result


def format_gcvx_to_fne(gcvx_data: Dict[str, Any], destination: str) -> Dict[str, Any]:
    """
    Fonction principale de formatage GCVX vers FNE-CDI

    Cette fonction reçoit les données déjà validées par GCVXDocumentData

    Args:
        gcvx_data: Données au format GCVX (dict validé)
        destination: ID de la destination (non utilisé actuellement)

    Returns:
        Données au format FNE-CDI
    """
    # Recréer l'objet Pydantic depuis le dict validé
    gcvx_obj = GCVXDocumentData(**gcvx_data)
    mapper = GCVXToFNEMapper()
    return mapper.map_invoice(gcvx_obj)
