"""
Client HTTP avec gestion des retries et timeouts
"""
import requests
import time
from app.models import APIRequest, APIResponse
from app.logger import get_logger

logger = get_logger(__name__)

class APIClient:
    def __init__(self, max_retries: int = 3, retry_delays: list = None):
        self.max_retries = max_retries
        self.retry_delays = retry_delays or [5, 30, 120]  # 5s, 30s, 2min
        self.session = requests.Session()
    
    def send_request(self, api_request: APIRequest, retry_count: int = 0) -> APIResponse:
        """
        Envoie une requête HTTP avec retry automatique
        """
        start_time = time.time()
        
        try:
            response = self.session.request(
                method=api_request.method,
                url=api_request.url,
                headers=api_request.headers,
                json=api_request.payload,
                timeout=api_request.timeout
            )
            
            duration_ms = (time.time() - start_time) * 1000
            
            # Succès (2xx)
            if 200 <= response.status_code < 300:
                try:
                    response_data = response.json()
                except:
                    response_data = {"raw": response.text}
                
                logger.info(
                    "request_success",
                    destination=api_request.destination,
                    status_code=response.status_code,
                    duration_ms=round(duration_ms, 2),
                    queue_id=api_request.queue_id
                )
                
                return APIResponse(
                    queue_id=api_request.queue_id,
                    destination=api_request.destination,
                    request_payload=api_request.payload,
                    status_code=response.status_code,
                    response_data=response_data,
                    duration_ms=duration_ms
                )
            
            # Erreur client (4xx) - ne pas retenter
            elif 400 <= response.status_code < 500:
                error_msg = f"Client error {response.status_code}: {response.text[:200]}"
                logger.warning(
                    "client_error",
                    destination=api_request.destination,
                    status_code=response.status_code,
                    error=error_msg,
                    queue_id=api_request.queue_id
                )
                
                return APIResponse(
                    queue_id=api_request.queue_id,
                    destination=api_request.destination,
                    request_payload=api_request.payload,
                    status_code=response.status_code,
                    error=error_msg,
                    duration_ms=duration_ms
                )
            
            # Erreur serveur (5xx) - retenter
            else:
                error_msg = f"Server error {response.status_code}: {response.text[:200]}"
                logger.error(
                    "server_error",
                    destination=api_request.destination,
                    status_code=response.status_code,
                    error=error_msg,
                    queue_id=api_request.queue_id
                )
                
                if retry_count < self.max_retries:
                    return self._retry_request(api_request, retry_count, error_msg)
                
                return APIResponse(
                    queue_id=api_request.queue_id,
                    destination=api_request.destination,
                    request_payload=api_request.payload,
                    status_code=response.status_code,
                    error=f"Max retries reached. {error_msg}",
                    duration_ms=duration_ms
                )
        
        except requests.exceptions.Timeout:
            duration_ms = (time.time() - start_time) * 1000
            error_msg = f"Timeout after {api_request.timeout}s"
            logger.error(
                "request_timeout",
                destination=api_request.destination,
                timeout=api_request.timeout,
                queue_id=api_request.queue_id
            )

            if retry_count < self.max_retries:
                return self._retry_request(api_request, retry_count, error_msg)

            return APIResponse(
                queue_id=api_request.queue_id,
                destination=api_request.destination,
                request_payload=api_request.payload,
                error=f"Max retries reached. {error_msg}",
                duration_ms=duration_ms
            )

        except requests.exceptions.ConnectionError as e:
            duration_ms = (time.time() - start_time) * 1000
            error_msg = f"Connection error: {str(e)[:200]}"
            logger.error(
                "connection_error",
                destination=api_request.destination,
                error=error_msg,
                queue_id=api_request.queue_id
            )

            if retry_count < self.max_retries:
                return self._retry_request(api_request, retry_count, error_msg)

            return APIResponse(
                queue_id=api_request.queue_id,
                destination=api_request.destination,
                request_payload=api_request.payload,
                error=f"Max retries reached. {error_msg}",
                duration_ms=duration_ms
            )

        except Exception as e:
            duration_ms = (time.time() - start_time) * 1000
            error_msg = f"Unexpected error: {str(e)[:200]}"
            logger.error(
                "unexpected_error",
                destination=api_request.destination,
                error=error_msg,
                queue_id=api_request.queue_id
            )

            return APIResponse(
                queue_id=api_request.queue_id,
                destination=api_request.destination,
                request_payload=api_request.payload,
                error=error_msg,
                duration_ms=duration_ms
            )
    
    def _retry_request(self, api_request: APIRequest, retry_count: int, _error_msg: str) -> APIResponse:
        """Gère le retry d'une requête"""
        retry_count += 1
        delay = self.retry_delays[min(retry_count - 1, len(self.retry_delays) - 1)]

        logger.info(
            "retrying_request",
            destination=api_request.destination,
            retry_count=retry_count,
            max_retries=self.max_retries,
            delay_seconds=delay,
            queue_id=api_request.queue_id
        )
        time.sleep(delay)
        
        return self.send_request(api_request, retry_count)
    
    def send_batch(self, requests: list[APIRequest]) -> list[APIResponse]:
        """Envoie plusieurs requêtes (séquentiel pour l'instant)"""
        responses = []
        for req in requests:
            response = self.send_request(req)
            responses.append(response)
        return responses