"""
Module générique d'envoi d'emails
Supporte Mailgun et Brevo (SendinBlue)
"""
from typing import Dict, Optional, Any, List, Tuple, Union
from pathlib import Path

from .mailgun_client import MailgunClient, MailgunConfig
from .brevo_client import BrevoClient, BrevoConfig
from .logger import get_logger

logger = get_logger("email_sender")


class EmailSender:
    """
    Gestionnaire générique d'envoi d'emails
    Supporte Mailgun et Brevo
    """

    def __init__(
        self,
        provider: str,
        api_key: str,
        from_email: str,
        from_name: str = "",
        domain: Optional[str] = None,
        **kwargs
    ):
        """
        Initialise l'envoyeur d'emails

        Args:
            provider: "mailgun" ou "brevo"
            api_key: Clé API du provider
            from_email: Email expéditeur
            from_name: Nom de l'expéditeur (optionnel)
            domain: Domaine (requis pour Mailgun uniquement)
        """
        self.provider = provider.lower()

        logger.info(
            "email_sender_init",
            provider=self.provider,
            from_email=from_email,
            from_name=from_name,
            domain=domain if domain else None
        )

        # Initialiser le client approprié
        if self.provider == "mailgun":
            if not domain:
                raise ValueError("Le domaine est requis pour Mailgun")
            self.client = MailgunClient(MailgunConfig(
                api_key=api_key,
                domain=domain,
                from_email=from_email,
                from_name=from_name,
                **kwargs
            ))
        elif self.provider == "brevo":

            self.client = BrevoClient(BrevoConfig(
                api_key=api_key,
                from_email=from_email,
                from_name=from_name,
                subject=kwargs.get("subject"),
                tags=kwargs.get("tags")
            ))
        else:
            logger.error("unsupported_provider", provider=provider)
            raise ValueError(f"Provider non supporté: {provider}")

    @classmethod
    def from_config(cls, config: Dict[str, Any]) -> "EmailSender":
        """
        Crée une instance depuis une configuration dict/TOML

        Args:
            config: Configuration avec structure:
                {
                    "email": {
                        "provider": "mailgun|brevo",
                        "mailgun": {"api_key": "...", "domain": "...", ...},
                        "brevo": {"api_key": "...", ...}
                    }
                }

        Returns:
            Instance de EmailSender
        """

        email_config = config.get("email", {})
        provider = email_config.get("provider", "mailgun")
        if email_config.get("from_email") is None:
            email_config["from_email"] = None

        # On charge d'abord les paramètres obligatoires et communs
        if provider not in ["mailgun", "brevo"]:
            raise ValueError(f"Provider non supporté dans la config: {provider}")

        return cls(provider, **email_config.get(provider, {}))


    def send(
        self,
        to_email: str,
        subject: str,
        body: str,
        body_type: str = "html",
        attachments: Optional[List[Tuple[str, bytes, str]]] = None,
        inline_images: Optional[Dict[str, bytes]] = None,
        cc: Optional[List[str]] = None,
        bcc: Optional[List[str]] = None,
    ) -> Dict[str, Any]:
        """
        Envoie un email générique

        Args:
            to_email: Adresse du destinataire
            subject: Sujet de l'email
            body: Corps de l'email (texte ou HTML selon body_type)
            body_type: "text" ou "html" (défaut: html)
            attachments: Liste de tuples (filename, content_bytes, mime_type)
            inline_images: Dict {cid: image_bytes} pour images inline (HTML uniquement)
            cc: Liste d'adresses en copie
            bcc: Liste d'adresses en copie cachée

        Returns:
            Réponse du provider (dict avec au moins message_id)

        Example:
            # Email HTML simple
            sender.send(
                to_email="user@example.com",
                subject="Test",
                body="<h1>Hello</h1>",
                body_type="html"
            )

            # Email texte avec pièce jointe
            sender.send(
                to_email="user@example.com",
                subject="Test",
                body="Hello world",
                body_type="text",
                attachments=[("doc.pdf", pdf_bytes, "application/pdf")]
            )

            # Email HTML avec image inline
            sender.send(
                to_email="user@example.com",
                subject="Test",
                body='<img src="cid:logo">',
                inline_images={"logo": logo_bytes}
            )
        """
        # Préparer les arguments selon le type de body
        kwargs = {
            "to_email": to_email,
            "subject": subject,
            "attachments": attachments,
            "inline_images": inline_images,
            "cc": cc,
            "bcc": bcc,
        }

        if body_type == "html":
            kwargs["html_body"] = body
        else:
            kwargs["text_body"] = body

        return self.client.send_email(**kwargs)

    def send_html(
        self,
        to_email: str,
        subject: str,
        html_body: str,
        text_body: Optional[str] = None,
        **kwargs
    ) -> Dict[str, Any]:
        """
        Raccourci pour envoyer un email HTML

        Args:
            to_email: Destinataire
            subject: Sujet
            html_body: Corps HTML
            text_body: Corps texte alternatif (optionnel)
            **kwargs: Autres arguments (attachments, inline_images, etc.)

        Returns:
            Réponse du provider
        """
        return self.client.send_email(
            to_email=to_email,
            subject=subject,
            html_body=html_body,
            text_body=text_body,
            **kwargs
        )

    def send_template(
        self,
        to_email: str,
        template_id: int,
        template_params: Dict[str, str],
        **kwargs
    ) -> Dict[str, Any]:
        """
        Envoie un email en utilisant un template (Brevo uniquement)

        Args:
            to_email: Destinataire
            template_id: ID du template
            template_params: Paramètres du template
            **kwargs: Autres arguments (cc, bcc)

        Returns:
            Réponse du provider

        Raises:
            NotImplementedError: Si le provider ne supporte pas les templates
        """
        if self.provider != "brevo":
            logger.warning(
                "send_template_not_supported",
                provider=self.provider,
                to_email=to_email,
                template_id=template_id
            )
            raise NotImplementedError(
                f"L'envoi via template n'est pas supporté par {self.provider}. "
                "Cette fonctionnalité est uniquement disponible avec Brevo."
            )

        logger.debug(
            "send_template_request",
            provider=self.provider,
            to_email=to_email,
            template_id=template_id
        )

        # logger.debug("send_template_params", template_params=template_params)

        return self.client.send_email_with_template(
            to_email=to_email,
            template_id=template_id,
            template_params=template_params,
            cc=kwargs.get("cc"),
            bcc=kwargs.get("bcc")
        )

    def send_text(
        self,
        to_email: str,
        subject: str,
        text_body: str,
        **kwargs
    ) -> Dict[str, Any]:
        """
        Raccourci pour envoyer un email texte

        Args:
            to_email: Destinataire
            subject: Sujet
            text_body: Corps texte
            **kwargs: Autres arguments (attachments, cc, bcc, etc.)

        Returns:
            Réponse du provider
        """
        return self.client.send_email(
            to_email=to_email,
            subject=subject,
            text_body=text_body,
            **kwargs
        )

    def send_file(
        self,
        to_email: str,
        subject: str,
        body: str,
        filename: str,
        file_content: bytes,
        mime_type: str = "application/octet-stream",
        body_type: str = "text"
    ) -> Dict[str, Any]:
        """
        Envoie un email avec une pièce jointe

        Args:
            to_email: Destinataire
            subject: Sujet
            body: Message d'accompagnement
            filename: Nom du fichier
            file_content: Contenu du fichier en bytes
            mime_type: Type MIME du fichier
            body_type: "text" ou "html"

        Returns:
            Réponse du provider
        """
        return self.send(
            to_email=to_email,
            subject=subject,
            body=body,
            body_type=body_type,
            attachments=[(filename, file_content, mime_type)]
        )

    def verify_configuration(self) -> bool:
        """
        Vérifie que la configuration email est valide

        Returns:
            True si la configuration est valide
        """
        logger.debug("verify_configuration", provider=self.provider)

        result = False
        if self.provider == "mailgun":
            result = self.client.verify_domain()
        elif self.provider == "brevo":
            result = self.client.verify_api_key()

        logger.info(
            "configuration_verified",
            provider=self.provider,
            is_valid=result
        )
        return result
