"""
Générateur de QR codes pour les liens de téléchargement de factures
Supporte deux backends:
- qrencode (utilitaire système) pour CentOS 6
- qrcode + Pillow (Python) pour les autres OS
"""
import io
import platform
import subprocess
import shutil
from typing import Optional


def _is_centos6() -> bool:
    """Détecte si on est sur CentOS 6"""
    try:
        with open('/etc/centos-release', 'r') as f:
            content = f.read()
            return 'CentOS release 6' in content or 'CentOS Linux release 6' in content
    except FileNotFoundError:
        return False


def _has_qrencode() -> bool:
    """Vérifie si qrencode est disponible"""
    return shutil.which('qrencode') is not None


def _generate_qrcode_qrencode(url: str, size: int = 300) -> bytes:
    """
    Génère un QR code avec qrencode (utilitaire système)

    Args:
        url: URL à encoder
        size: Taille approximative en pixels (divisée par 30 pour box_size)

    Returns:
        Bytes du QR code en format PNG
    """
    # Calculer box_size approximatif (qrencode génère ~33x33 modules par défaut)
    # Pour obtenir ~300px, on utilise box_size = 300/33 ≈ 9
    box_size = max(1, size // 33)

    result = subprocess.run(
        ['qrencode', '-s', str(box_size), '-m', '2', '-o', '-', url],
        capture_output=True,
        check=True
    )
    return result.stdout


def _generate_qrcode_pillow(url: str, size: int = 300) -> bytes:
    """
    Génère un QR code avec qrcode + Pillow (Python)

    Args:
        url: URL à encoder
        size: Taille en pixels

    Returns:
        Bytes du QR code en format PNG
    """
    import qrcode
    from PIL import Image

    # Créer le QR code
    qr = qrcode.QRCode(
        version=1,
        error_correction=qrcode.constants.ERROR_CORRECT_L,
        box_size=10,
        border=4,
    )

    qr.add_data(url)
    qr.make(fit=True)

    # Générer l'image
    img = qr.make_image(fill_color="black", back_color="white")

    # Redimensionner si nécessaire
    if size != 300:
        img = img.resize((size, size), Image.Resampling.LANCZOS)

    # Convertir en bytes PNG
    buffer = io.BytesIO()
    img.save(buffer, format='PNG')
    buffer.seek(0)

    return buffer.getvalue()


def generate_qrcode(url: str, size: int = 300) -> bytes:
    """
    Génère un QR code à partir d'une URL
    Choisit automatiquement le backend approprié:
    - qrencode sur CentOS 6
    - qrcode + Pillow sur les autres OS

    Args:
        url: URL à encoder dans le QR code
        size: Taille du QR code en pixels (défaut: 300x300)

    Returns:
        Bytes du QR code en format PNG

    Raises:
        RuntimeError: Si aucun backend disponible
    """
    # Sur CentOS 6, utiliser qrencode si disponible
    if _is_centos6() and _has_qrencode():
        return _generate_qrcode_qrencode(url, size)

    # Sinon, essayer Pillow
    try:
        return _generate_qrcode_pillow(url, size)
    except ImportError:
        # Si Pillow pas disponible, essayer qrencode en fallback
        if _has_qrencode():
            return _generate_qrcode_qrencode(url, size)
        else:
            raise RuntimeError(
                "Aucun backend de génération de QR code disponible. "
                "Installez 'pip install qrcode pillow' ou 'yum install qrencode'"
            )


def generate_qrcode_base64(url: str, size: int = 300) -> str:
    """
    Génère un QR code en base64 pour inclusion dans un email HTML

    Args:
        url: URL à encoder
        size: Taille en pixels

    Returns:
        String base64 du QR code
    """
    import base64

    qr_bytes = generate_qrcode(url, size)
    return base64.b64encode(qr_bytes).decode('utf-8')
