"""
Client Mailgun pour l'envoi d'emails
"""
import requests
import time
from typing import Dict, List, Optional, Tuple
from dataclasses import dataclass

from .logger import get_logger, log_email_sent, log_email_error, log_api_call

logger = get_logger("email_sender.mailgun")


@dataclass
class MailgunConfig:
    """Configuration Mailgun"""
    api_key: str
    domain: str
    from_email: str
    from_name: str = "GCV Gestion"


class MailgunClient:
    """Client pour l'API Mailgun"""

    def __init__(self, config: MailgunConfig):
        """
        Initialise le client Mailgun

        Args:
            config: Configuration Mailgun
        """
        self.config = config
        self.base_url = f"https://api.mailgun.net/v3/{config.domain}"

    def send_email(
        self,
        to_email: str,
        subject: str,
        html_body: str,
        text_body: Optional[str] = None,
        attachments: Optional[List[Tuple[str, bytes, str]]] = None,
        inline_images: Optional[Dict[str, bytes]] = None,
        cc: Optional[List[str]] = None,
        bcc: Optional[List[str]] = None,
    ) -> Dict:
        """
        Envoie un email via Mailgun

        Args:
            to_email: Adresse du destinataire
            subject: Sujet de l'email
            html_body: Corps HTML de l'email
            text_body: Corps texte brut (optionnel)
            attachments: Liste de tuples (filename, content, mime_type)
            inline_images: Dict {cid: image_bytes} pour les images inline
            cc: Liste d'adresses en copie
            bcc: Liste d'adresses en copie cachée

        Returns:
            Réponse de l'API Mailgun

        Raises:
            requests.exceptions.RequestException: En cas d'erreur API
        """
        # Préparer les données
        data = {
            "from": f"{self.config.from_name} <{self.config.from_email}>",
            "to": to_email,
            "subject": subject,
            "html": html_body,
        }

        if text_body:
            data["text"] = text_body

        if cc:
            data["cc"] = cc

        if bcc:
            data["bcc"] = bcc

        # Préparer les fichiers (attachments et inline images)
        files = []

        if attachments:
            for filename, content, mime_type in attachments:
                files.append(
                    ("attachment", (filename, content, mime_type))
                )

        if inline_images:
            for cid, image_bytes in inline_images.items():
                files.append(
                    ("inline", (cid, image_bytes, "image/png"))
                )

        # Logger la requête
        logger.debug(
            "mailgun_send_email_request",
            to_email=to_email,
            subject=subject,
            has_attachments=bool(attachments),
            has_inline_images=bool(inline_images),
            has_cc=bool(cc),
            has_bcc=bool(bcc)
        )

        # Envoyer la requête avec mesure du temps
        start_time = time.time()
        try:
            response = requests.post(
                f"{self.base_url}/messages",
                auth=("api", self.config.api_key),
                data=data,
                files=files if files else None,
                timeout=30
            )
            duration_ms = (time.time() - start_time) * 1000

            # Logger l'appel API
            log_api_call(
                provider="mailgun",
                endpoint="/messages",
                method="POST",
                status_code=response.status_code,
                duration_ms=duration_ms
            )

            # En cas d'erreur, logger et lever l'exception
            if not response.ok:
                log_email_error(
                    provider="mailgun",
                    to_email=to_email,
                    error=f"{response.status_code} {response.reason}",
                    error_code=str(response.status_code),
                    subject=subject
                )
                response.raise_for_status()

            result = response.json()

            # Logger le succès
            log_email_sent(
                provider="mailgun",
                to_email=to_email,
                subject=subject,
                message_id=result.get("id"),
                duration_ms=duration_ms
            )

            return result

        except requests.exceptions.RequestException as e:
            duration_ms = (time.time() - start_time) * 1000
            log_email_error(
                provider="mailgun",
                to_email=to_email,
                error=str(e),
                subject=subject,
                duration_ms=duration_ms
            )
            raise

    def verify_domain(self) -> bool:
        """
        Vérifie que le domaine est correctement configuré

        Returns:
            True si le domaine est vérifié
        """
        try:
            logger.debug("mailgun_verify_domain", domain=self.config.domain)
            response = requests.get(
                f"{self.base_url}",
                auth=("api", self.config.api_key),
                timeout=10
            )
            is_valid = response.status_code == 200
            logger.info(
                "mailgun_domain_verified",
                domain=self.config.domain,
                is_valid=is_valid,
                status_code=response.status_code
            )
            return is_valid
        except requests.exceptions.RequestException as e:
            logger.error("mailgun_verify_domain_error", domain=self.config.domain, error=str(e))
            return False
