#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Générateur de factures à partir de données JSON - Version avec gestion des empreintes
"""

import json
from datetime import datetime
from pathlib import Path
from dataclasses import dataclass, field
from typing import Iterator, List, Tuple, Optional
from reportlab.lib.pagesizes import A4
from reportlab.lib.units import mm
from reportlab.pdfgen import canvas
from reportlab.platypus import Paragraph
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.lib.enums import TA_LEFT
import barcode
from barcode.writer import ImageWriter
import qrcode
import tempfile
import os
import typer

bgColor = (0.85, 0.85, 0.85)

# ========== CLASSES D'EMPREINTE ==========

@dataclass
class BoundingBox:
    """Représente l'empattement d'un élément sur la page"""
    x: float  # Position X (bas-gauche)
    y: float  # Position Y (bas-gauche)
    width: float
    height: float
    label: str = "Element"  # Pour identifier l'élément
    
    @property
    def x_max(self) -> float:
        """Coordonnée X maximale (droite)"""
        return self.x + self.width
    
    @property
    def y_max(self) -> float:
        """Coordonnée Y maximale (haut)"""
        return self.y + self.height
    
    @property
    def center_x(self) -> float:
        """Centre X"""
        return self.x + self.width / 2
    
    @property
    def center_y(self) -> float:
        """Centre Y"""
        return self.y + self.height / 2
    
    def overlaps(self, other: 'BoundingBox') -> bool:
        """Vérifie si cet élément chevauche un autre"""
        return not (
            self.x_max <= other.x or  # Complètement à gauche
            self.x >= other.x_max or  # Complètement à droite
            self.y_max <= other.y or  # Complètement en bas
            self.y >= other.y_max     # Complètement en haut
        )
    
    def __repr__(self) -> str:
        return f"BoundingBox({self.label}: x={self.x:.1f}, y={self.y:.1f}, w={self.width:.1f}, h={self.height:.1f})"


class LayoutManager:
    """Gestionnaire de layout pour détecter les chevauchements"""
    
    def __init__(self):
        self.elements: List[BoundingBox] = []
    
    def add(self, bbox: BoundingBox) -> bool:
        """
        Ajoute un élément et vérifie les chevauchements
        Retourne True si ajouté sans chevauchement, False sinon
        """
        overlapping = self.check_overlaps(bbox)
        self.elements.append(bbox)
        return len(overlapping) == 0
    
    def check_overlaps(self, bbox: BoundingBox) -> List[BoundingBox]:
        """Retourne la liste des éléments qui chevauchent avec bbox"""
        return [elem for elem in self.elements if bbox.overlaps(elem)]
    
    def report_overlaps(self) -> List[str]:
        """Génère un rapport des chevauchements détectés"""
        messages = []
        checked = set()
        
        for i, elem1 in enumerate(self.elements):
            for j, elem2 in enumerate(self.elements[i+1:], start=i+1):
                pair = (i, j)
                if pair not in checked and elem1.overlaps(elem2):
                    messages.append(
                        f"ATTENTION: Chevauchement detecte entre '{elem1.label}' et '{elem2.label}'"
                    )
                    checked.add(pair)
        
        return messages


# ========== CONTEXTES ==========

@dataclass
class PageContext:
    """Contexte contenant les informations sur la page et le canvas"""
    canvas: canvas.Canvas
    width: float
    height: float
    margin_left: float
    margin_right: float
    margin_top: float
    margin_bottom: float
    logo_path: str = None
    debug_logo: bool = False
    layout_manager: LayoutManager = None
    
    def __post_init__(self):
        if self.layout_manager is None:
            self.layout_manager = LayoutManager()
    
    @property
    def textwidth(self):
        """Largeur utilisable de la page"""
        return self.width - self.margin_left - self.margin_right

    @property
    def textheight(self):
        """Largeur utilisable de la page"""
        return self.height - self.margin_top - self.margin_bottom
       

@dataclass
class DataContext:
    """Contexte contenant les données de la facture"""
    data: dict

    _FAC_TYP_BSM = {
        1: "BIEN",
        2: "SERVICE",
        3: "SERVICE"
    }

    @property
    def emetteur_logo_path(self):
        return self.data.get('emetteur', {}).get('logo_path', None)

    @property
    def fac_num(self):
        return self.data.get('fac_num', '')
    
    @property
    def fac_type(self):
        return self.data.get('fac_type', 'FACTURE')
    
    @property
    def fac_date(self):
        return self.data.get('fac_date', '')
    
    @property
    def emetteur(self):
        return self.data.get('emetteur', {})
    
    @property
    def client(self):
        return self.data.get('client', {})
    
    @property
    def adresses_client(self):
        return self.client.get('adresses', {})

    @property
    def client_facturation(self):
        return self.adresses_client.get('facturation', {})
    
    @property
    def client_livraison(self):
        return self.adresses_client.get('livraison', {})

    @property
    def client_adresse_actuelle(self):
        return self.adresses_client.get('actuelle', {})

    @property
    def emetteur_version(self):
        return self.emetteur.get('version', 'v2')

    @property
    def emetteur_pays(self):
        return self.emetteur.get('pays', '')

    @property
    def emetteur_short_pays(self):
        return self.emetteur.get('short_pays', "FR")

    @property
    def client_short_pays(self):
        return self.client_facturation.get("short_pays", "FR")

    @property
    def vendeur(self):
        return self.data.get('vendeur', {})
    
    @property
    def articles(self):
        return self.data.get('articles', [])
    
    @property
    def pied(self):
        return self.data.get('pied', {})
    
    @property
    def reglements(self):
        return self.data.get('reglements', [])
    
    @property
    def emetteur_legacy(self):
        return self.data.get("emetteur_legacy_header", [])
    
    @property
    def fac_type_bsm(self):
        return self._FAC_TYP_BSM.get(
            self.data.get("type_BSM", 1),
            ""
            ).title()


# ========== UTILITAIRES ==========
def draw_anchor(c, x, y, box_x=0, box_y=0):
    # Ligne de référence pour visualiser
    c.setStrokeColorRGB(1, 0, 0)
    if box_x * box_y != 0:
        c.line(x, y - 5, box_x, box_y + 15)
    c.circle(x, y, 2, fill=1)
    c.setStrokeColorRGB(0, 0, 0)

def generate_barcode(code, barcode_type='code128'):
    """Génère un code-barres"""
    try:
        CODE = barcode.get_barcode_class(barcode_type)
        code_barres = CODE(code, writer=ImageWriter())
        temp_file = tempfile.NamedTemporaryFile(delete=False, suffix='.png')
        temp_filename = temp_file.name.replace('.png', '')
        temp_file.close()
        code_barres.save(temp_filename, {'write_text': False, 'module_height': 15, 'module_width': 0.3})
        final_path = temp_filename + '.png'
        return final_path
    except Exception as e:
        print(f"Erreur barcode: {e}")
        return None


def formater_date(date_str):
    """Convertit une date du format YYYYMMDD HHMMSS vers DD/MM/YYYY"""
    try:
        if len(date_str) >= 8:
            annee = date_str[0:4]
            mois = date_str[4:6]
            jour = date_str[6:8]
            return f"{jour}/{mois}/{annee}"
    except:
        return date_str
    return date_str


def formater_heure(date_str):
    """Extrait l'heure du format YYYYMMDD HHMMSS vers HH:MM"""
    try:
        if len(date_str) >= 14:
            heure = date_str[9:11]
            minute = date_str[11:13]
            return f"{heure}H{minute}"
    except:
        return ""
    return ""


# ========== FONCTIONS DE DESSIN ==========

def draw_logo(page_ctx: PageContext) -> BoundingBox:
    """Dessine le logo ou un placeholder et retourne son empattement"""
    c = page_ctx.canvas
    logo_width = 200
    logo_height = 66
    logo_x = page_ctx.margin_left
    logo_y = page_ctx.height - page_ctx.margin_top - logo_height

    if page_ctx.logo_path and os.path.exists(page_ctx.logo_path):
        c.drawImage(page_ctx.logo_path, logo_x, logo_y, 
                   width=logo_width, height=logo_height,
                   preserveAspectRatio=True, mask='auto')
        
        if page_ctx.debug_logo:
            c.setStrokeColorRGB(1, 0, 0)
            c.setLineWidth(0.5)
            c.rect(logo_x, logo_y, logo_width, logo_height)
            c.setStrokeColorRGB(0, 0, 0)
            c.setLineWidth(1)
    else:
        c.setLineWidth(0.5)
        c.rect(logo_x, logo_y, logo_width, logo_height)
        c.line(logo_x, logo_y, logo_x + logo_width, logo_y + logo_height)
        c.line(logo_x + logo_width, logo_y, logo_x, logo_y + logo_height)
        c.setLineWidth(1)
        c.setFont("Helvetica", 9)
        c.drawCentredString(logo_x + logo_width/2, logo_y + logo_height/2, "LOGO")
    
    bbox = BoundingBox(logo_x, logo_y, logo_width, logo_height, "Logo")
    page_ctx.layout_manager.add(bbox)
    return bbox


def draw_numero_piece(page_ctx: PageContext, data_ctx: DataContext, with_border=False) -> BoundingBox:
    """Dessine le cadre du numéro de facture et retourne son empattement"""
    c = page_ctx.canvas
    box_width = 180
    box_height = 25
    box_x = page_ctx.width - page_ctx.margin_right
    box_y = page_ctx.height - page_ctx.margin_top - 22
    
    if with_border:
        c.setLineWidth(0.5)
        c.rect(box_x, box_y, box_width, box_height)
    c.setFont("Helvetica-Bold", 13)
    c.drawRightString(box_x, box_y, 
                       f"{data_ctx.fac_type.capitalize()} N° {data_ctx.fac_num}")
    if with_border:
        c.setLineWidth(1)
    
    bbox = BoundingBox(box_x, box_y, box_width, box_height, "Numero_Piece")
    page_ctx.layout_manager.add(bbox)
    return bbox


def draw_client_facturation(page_ctx: PageContext, data_ctx: DataContext, x:float, y:float, w:float, h:float) -> BoundingBox:
    """Dessine les informations client et retourne son empattement"""
    c = page_ctx.canvas
    box_x, box_y, box_width, box_height = x, y, w, h

    line_height = 12

    # draw_anchor(c, box_x, box_y)
    c.setLineWidth(0.5)
    c.rect(box_x, box_y, box_width, box_height)
    c.setLineWidth(1)
    
    client = data_ctx.client
    client_fac = data_ctx.client_facturation

    c.setFont("Helvetica-Bold", 9)
    c.drawString(box_x + 5, box_y + 83, "Facturé à")

    y_cli_adr_fac = box_y + 68
    c.setFont("Helvetica-Bold", 10)
    c.drawString(box_x + 5, y_cli_adr_fac, client_fac.get('raison_sociale', ''))
    y_cli_adr_fac -= line_height
    c.setFont("Helvetica", 9)
    for i in range(3):
        c.drawString(box_x + 5, y_cli_adr_fac, client_fac.get(f'ligne{i+1}', ''))
        y_cli_adr_fac -= line_height

    ville_cp = ""
    if client_fac.get('code_postal'):
        ville_cp = client_fac.get('code_postal', '') + " "
    ville_cp += client_fac.get('ville', '')
    c.drawString(box_x + 5, y_cli_adr_fac, ville_cp)
    y_cli_adr_fac -= line_height

    if isPro := (client.get("type","")=="professionnel"):
        if tva_intra := client_fac.get("tva_intra",""):
            c.setFont("Helvetica", 8)
            c.drawString(box_x + 5, y_cli_adr_fac, "TVA Intra :")
            c.drawString(box_x + 45, y_cli_adr_fac, tva_intra)
            y_cli_adr_fac -= line_height
            c.setFont("Helvetica", 9)
        #     if data_ctx.client_short_pays == "FR" and data_ctx.emetteur_short_pays == "FR":
        #         c.drawString(box_x + 5, box_y + 8, "TVA Intra :")

        #     if data_ctx.client_short_pays == "CDI" and data_ctx.emetteur_short_pays == "CDI":
        #         c.setFont("Helvetica-Bold", 10)
        #         c.drawString(box_x + 5, box_y + 68, tva_intra)
        #         c.setFont("Helvetica", 9)

    bbox = BoundingBox(box_x, box_y, box_width, box_height, "Info_Client")
    # page_ctx.layout_manager.add(bbox)
    return bbox

def draw_client_livraison(page_ctx: PageContext, data_ctx: DataContext, x:float, y:float, w:float, h:float) -> BoundingBox:
    """Dessine les informations client et retourne son empattement"""
    c = page_ctx.canvas
    box_x, box_y, box_width, box_height = x, y, w, h
    
    c.setLineWidth(0.5)
    c.rect(box_x, box_y, box_width, box_height)
    c.setLineWidth(1)
    
    client = data_ctx.client_livraison
    
    c.setFont("Helvetica-Bold", 9)
    c.drawString(box_x + 5, box_y + 83, "Livré à")

    c.setFont("Helvetica-Bold", 10)
    c.drawString(box_x + 5, box_y + 68, client.get('raison_sociale', ''))
    c.setFont("Helvetica", 9)
    c.drawString(box_x + 5, box_y + 55, client.get('adresse', ''))
    
    ville_cp = ""
    if client.get('code_postal'):
        ville_cp = client.get('code_postal', '') + " "
    ville_cp += client.get('ville', '')
    c.drawString(box_x + 5, box_y + 43, ville_cp)
    
    bbox = BoundingBox(box_x, box_y, box_width, box_height, "Info_Client")
    # page_ctx.layout_manager.add(bbox)
    return bbox

def draw_entete_client(page_ctx: PageContext, data_ctx: DataContext) -> BoundingBox:
    box_width = 180
    box_height = 80
    client_info_width = 180
    client_info_height = 80

    client_info_x = page_ctx.width - page_ctx.margin_right - client_info_width
    client_info_y = (page_ctx.height - page_ctx.margin_top - 90) - client_info_height

    livraison_width, livraison_height = client_info_width, client_info_height

    livraison_x = client_info_x - livraison_width
    livraison_y = client_info_y

    client_bbox = draw_client_facturation(page_ctx, data_ctx, client_info_x, client_info_y, box_width, box_height)
    client_livraison_bbox = draw_client_livraison(page_ctx, data_ctx, livraison_x, livraison_y, livraison_width, livraison_height)

    bbox = BoundingBox(
        min(client_bbox.x,client_livraison_bbox.x),
        min(client_bbox.y,client_livraison_bbox.y),
        client_bbox.width + client_livraison_bbox.width,
        client_bbox.height + client_livraison_bbox.height,
        "Entete_Client"
    )
    page_ctx.layout_manager.add(bbox)
    return bbox


def draw_emetteur_legacy_header(page_ctx: PageContext, data_ctx: DataContext, y_start: float) -> Tuple[float, BoundingBox]:
    c = page_ctx.canvas
    emetteur_legacy = data_ctx.emetteur_legacy
    
    x = page_ctx.margin_left
    y = y_start
    y_top = y
    
    c.setFont("Helvetica-Bold", 9)
    c.drawString(x, y, emetteur_legacy[0].strip())
    y -= 12
    c.setFont("Helvetica", 8)
    for line in emetteur_legacy[1:]:
        c.drawString(x, y, line.strip())
        y -= 12
    
    width = 200  # Estimation de la largeur du bloc texte
    bbox = BoundingBox(x, y, width, abs(y_start-y), "Info_Emetteur")
    page_ctx.layout_manager.add(bbox)
    
    return y - 25, bbox


def draw_emetteur_info(page_ctx: PageContext, data_ctx: DataContext, y_start: float) -> Tuple[float, BoundingBox]:
    """Dessine les informations de l'émetteur et retourne (y_final, empattement)"""
    c = page_ctx.canvas
    emetteur = data_ctx.emetteur
    client = data_ctx.client_facturation
    
    x = page_ctx.margin_left
    y = y_start
    y_top = y
    
    c.setFont("Helvetica-Bold", 9)
    c.drawString(x, y, emetteur.get('raison_sociale', ''))
    y -= 12
    c.drawString(x, y, emetteur.get('adresse', ''))
    y -= 12
    
    ville_emit = emetteur.get('code_postal', '') + " " + emetteur.get('ville', '')
    c.drawString(x, y, ville_emit)
    y -= 12
    
    c.setFont("Helvetica", 8)
    
    # Téléphone
    tel_info = ""
    for art in data_ctx.articles:
        if art.get('code_article') == '$':
            tel_info = art.get('description', '')
            break
    if tel_info:
        c.drawString(x, y, tel_info)
    y -= 10
    
    c.drawString(x, y, f"Siret : {emetteur.get('siret', '')}")
    y -= 10
    
    form_jur = emetteur.get('form_juridique', '')
    capital = emetteur.get('capital', '')
    c.drawString(x, y, f"{form_jur} - Capital {capital}")
    y -= 10
    
    if client.get('email'):
        c.drawString(x, y, f"Email : {client.get('email', '')}")
    
    y_bottom = y
    height = y_top - y_bottom
    width = 200  # Estimation de la largeur du bloc texte
    
    bbox = BoundingBox(x, y_bottom, width, height, "Info_Emetteur")
    page_ctx.layout_manager.add(bbox)
    
    return y - 25, bbox


def draw_info_facturation_v2(page_ctx: PageContext, data_ctx: DataContext, y_start: float) -> Tuple[float, BoundingBox]:
    """Dessine les informations de facturation et retourne (y_final, empattement)"""
    c = page_ctx.canvas
    info_y = y_start
    y_top = info_y
    row_height = 12
    
    c.setLineWidth(0.5)
    c.setFont("Helvetica", 8)


    date_label_width = 25
    date_value_width = 85
    client_label_width = 25
    client_value_width = 100
    caisse_label_width = 30
    caisse_value_width = 15
    vendeur_label_width = 35
    vendeur_value_width = 120
    
    date_formattee = formater_date(data_ctx.fac_date)
    heure_formattee = formater_heure(data_ctx.fac_date)
    
    # === LIGNE 1 ===
    x_pos = page_ctx.margin_left
    
    def draw_title_box(c, x_pos, title, width):
        c.setFillColorRGB(*bgColor)
        c.rect(x_pos, info_y - row_height, width, row_height, fill=1, stroke=1)
        c.setFillColorRGB(0, 0, 0)
        c.drawString(x_pos + 2, info_y - 8, title)
        x_pos += width
        return x_pos

    def draw_value_box(c, x_pos, value, width):
        c.rect(x_pos, info_y - row_height, width, row_height, fill=0, stroke=1)
        c.drawString(x_pos + 2, info_y - 8, value)
        x_pos += width
        return x_pos

    x_pos = draw_title_box(c, x_pos, "Date", date_label_width)
    x_pos = draw_value_box(c, x_pos, f"{date_formattee} {heure_formattee}", date_value_width)
        
    x_pos = draw_title_box(c, x_pos, "Client", client_label_width)
    cardnumber = data_ctx.client.get("cardnumber", "")
    if cardnumber:
        x_pos = draw_value_box(c, x_pos, cardnumber, client_value_width)
    else:
        value = data_ctx.client.get('code', '')
        x_pos = draw_value_box(c, x_pos, value, client_value_width)

    x_pos = draw_title_box(c, x_pos, "Caisse", caisse_label_width)
    
    c.rect(x_pos, info_y - row_height, caisse_value_width , row_height, fill=0, stroke=1)
    c.setFont("Helvetica", 7)
    c.drawString(x_pos + 2, info_y - 8, f"{data_ctx.data.get('num_caisse', '')}")
    c.setFont("Helvetica", 8)
    x_pos += caisse_value_width

    x_pos = draw_title_box(c, x_pos, "Vendeur", vendeur_label_width)
    
    vendeur = data_ctx.vendeur
    remaining_width = page_ctx.textwidth - (x_pos - page_ctx.margin_left)
    c.rect(x_pos, info_y - row_height, remaining_width, row_height, fill=0, stroke=1)
    c.setFont("Helvetica", 7)
    c.drawString(x_pos + 2, info_y - 8, f"{vendeur.get('nom', '')} ({vendeur.get('code', '')})")
    c.setFont("Helvetica", 8)
    x_pos += vendeur_value_width


    current_page = 10
    total_page = 99
    page_width = 8 * mm
    x_pos = page_ctx.width - page_ctx.margin_right - 2*page_width
    x_pos = draw_title_box(c, x_pos, "Page", page_width)
    c.setFont("Helvetica", 7)
    c.drawString(x_pos + 2, info_y - 8, f"{current_page}/{total_page}")
    c.setFont("Helvetica", 8)
    # x_pos += vendeur_value_width
    
    
    # === LIGNE 2 ===
    info_y -= row_height
    x_pos = page_ctx.margin_left
    c.rect(x_pos, info_y - row_height, page_ctx.textwidth, row_height, fill=0, stroke=1)
    x_pos = draw_title_box(c, x_pos, "Dépôt", date_label_width)
    depot = vendeur.get('point_de_vente', '')
    x_pos = draw_value_box(c, x_pos, depot, date_value_width)
    x_pos = draw_title_box(c, x_pos, "TVA", date_label_width)
    x_pos = draw_value_box(c, x_pos, f"XXXXX", date_value_width)
    x_pos = draw_title_box(c, x_pos, "Type", date_label_width)
    x_pos = draw_value_box(c, x_pos, data_ctx.fac_type_bsm, 15*mm)


    # # === LIGNE 3 ===
    # info_y -= row_height
    # x_pos = page_ctx.margin_left
    # c.rect(x_pos, info_y - row_height, page_ctx.textwidth, row_height, fill=0, stroke=1)
    # x_pos = draw_title_box(c, x_pos, "Dépôt", date_label_width)
    # x_pos = draw_value_box(c, x_pos, f"SIÈGE", date_value_width)
    # x_pos = draw_title_box(c, x_pos, "TVA", date_label_width)
    # x_pos = draw_value_box(c, x_pos, f"sur encaissement", date_value_width)
    # x_pos = draw_title_box(c, x_pos, "Type", date_label_width)
    # x_pos = draw_value_box(c, x_pos, f"Service", 15*mm)


    # === LIGNE 4 ===
    info_y -= row_height + 2
    x_pos = page_ctx.margin_left
    # c.rect(x_pos, info_y - row_height, page_ctx.textwidth, row_height, fill=0, stroke=1)

    # === LIGNE 5 ===
    info_y -= row_height + 2
    x_pos = page_ctx.margin_left
    # c.rect(x_pos, info_y - row_height, page_ctx.textwidth, row_height, fill=0, stroke=1)

    # Commentaires (si présents) - Utilisation de Paragraph pour gestion automatique multiligne
    commentaires = data_ctx.pied.get('commentaires', [])
    if commentaires:
        comment_text = " - ".join(commentaires)

        # Créer un style de paragraphe personnalisé
        style = ParagraphStyle(
            'CommentStyle',
            fontName='Helvetica',
            fontSize=8,
            leading=10,  # Espacement entre les lignes
            alignment=TA_LEFT,
            leftIndent=0,
            rightIndent=0,
        )

        # Créer le paragraphe
        para = Paragraph(comment_text, style)

        # Calculer la hauteur nécessaire
        max_width = page_ctx.textwidth
        w, h = para.wrap(max_width, 2 * row_height)  # Limite à 2 lignes max

        # Dessiner le paragraphe
        para.drawOn(c, page_ctx.margin_left, info_y - h)

        # Ajuster info_y en fonction de la hauteur utilisée
        info_y -= h

    # # === LIGNE 5 ===
    # info_y -= row_height
    # x_pos = page_ctx.margin_left
    
    # livraison = data_ctx.client_livraison
    
    # c.setFillColorRGB(*bgColor)
    # c.rect(x_pos, info_y - row_height, 45, row_height, fill=1, stroke=1)
    # c.setFillColorRGB(0, 0, 0)
    # c.setFont("Helvetica", 7)
    # c.drawString(x_pos + 2, info_y - 8, "Adresse Livraison")
    # c.setFont("Helvetica", 8)
    # x_pos += 45
    
    # c.rect(x_pos, info_y - row_height, 150, row_height, fill=0, stroke=1)
    # c.setFont("Helvetica", 7)
    # c.drawString(x_pos + 2, info_y - 8, livraison.get('adresse', ''))
    # c.setFont("Helvetica", 8)
    # x_pos += 150
    
    # c.setFillColorRGB(*bgColor)
    # c.rect(x_pos, info_y - row_height, 35, row_height, fill=1, stroke=1)
    # c.setFillColorRGB(0, 0, 0)
    # c.drawString(x_pos + 2, info_y - 8, "Dt Livr.")
    # x_pos += 35
    
    # c.rect(x_pos, info_y - row_height, 35, row_height, fill=0, stroke=1)
    # c.drawString(x_pos + 2, info_y - 8, date_formattee)
    # x_pos += 35
    
    # c.setFillColorRGB(*bgColor)
    # c.rect(x_pos, info_y - row_height, 50, row_height, fill=1, stroke=1)
    # c.setFillColorRGB(0, 0, 0)
    # c.setFont("Helvetica", 7)
    # c.drawString(x_pos + 2, info_y - 8, "Code service")
    # c.setFont("Helvetica", 8)
    # x_pos += 50
    
    # c.rect(x_pos, info_y - row_height, 50, row_height, fill=0, stroke=1)
    # x_pos += 50
    
    # c.setFillColorRGB(*bgColor)
    # c.rect(x_pos, info_y - row_height, 60, row_height, fill=1, stroke=1)
    # c.setFillColorRGB(0, 0, 0)
    # c.setFont("Helvetica", 7)
    # c.drawString(x_pos + 2, info_y - 8, "Référence marché")
    # c.setFont("Helvetica", 8)
    # x_pos += 60
    
    # remaining_width = page_ctx.textwidth - (x_pos - page_ctx.margin_left)
    # c.rect(x_pos, info_y - row_height, remaining_width, row_height, fill=0, stroke=1)
    
    # info_y -= row_height
    
    # # Pièce origine
    # piece_orig = data_ctx.data.get('piece_origine', '')
    # if piece_orig:
    #     c.setFont("Helvetica", 8)
    #     c.drawString(page_ctx.margin_left, info_y - 8, f"CF N°/REF : {piece_orig}")
    #     info_y -= row_height
    
    c.setLineWidth(1)
    
    y_bottom = info_y
    height = y_top - y_bottom
    
    bbox = BoundingBox(page_ctx.margin_left, y_bottom, page_ctx.textwidth, height, "Info_Facturation")
    page_ctx.layout_manager.add(bbox)
    
    return info_y - 10, bbox

@dataclass
class ArticleColumn:
    header: str
    width: float
    align: str  # 'left', 'right', 'center'
    name: str
    x: float = 0 

@dataclass
class ArticleTable:
    page_ctx: PageContext
    _columns: List[ArticleColumn] = field(default_factory=list)
    _x_offset: float = 0

    def add_column(self, header: str, width: float, align: str, name: str):
        if self._x_offset == 0:
            self._x_offset = self.page_ctx.margin_left

        if align not in ('left', 'right', 'center'):
            raise ValueError("Align must be 'left', 'right', or 'center'")
        
        x_offset = self._x_offset + width if align == 'right' else self._x_offset
        self._columns.append(ArticleColumn(header, width, align, name, x_offset))
        self._x_offset += width

    def get_headers(self) -> List[str]:
        return [col.header for col in self._columns]

    @property
    def total_width(self) -> float:
        return sum(col.width for col in self._columns)

    def __getitem__(self, name: str) -> ArticleColumn:
        for col in self._columns:
            if col.name == name:
                return col
        return None

    def __iter__(self) -> Iterator[ArticleColumn]:
        return iter(self._columns)
    
    def __len__(self) -> float:
        return len(self._columns)

def draw_articles_table(
        page_ctx: PageContext,
        data_ctx: DataContext,
        y_start: float,
        with_lines=True) -> Tuple[float, BoundingBox]:
    """Dessine le tableau des articles et retourne (y_final, empattement)"""
    c = page_ctx.canvas
    y = y_start
    y_top = y
    
    c.setLineWidth(0.5)
    
    article_table = ArticleTable(page_ctx=page_ctx)
    headers = [
        ("Référence"  ,  22*mm, 'left', 'code_article'),
        ("Désignation",  82*mm, 'left', 'description'),
        ("P.U. Net HT",  18*mm, 'right', 'pubase_ht'),
        ("Rem (%)"    ,  12*mm, 'right', 'tx_remise'),
        ("P.U. HT"    ,  18*mm, 'right', 'pu_ht'),
        ("Qté"        ,  12*mm, 'right', 'quantite'),
        ("Montant HT" ,  20*mm, 'right', 'montant_ht'),
        ("T"          ,   6*mm, 'center', 'tva_code'),
    ]
    for header in headers:
        article_table.add_column(*header)

    col_widths = [col.width for col in article_table._columns]
    col_x = [page_ctx.margin_left] 
    for w in col_widths[:-1]:
        col_x.append(col_x[-1] + w)
    
    # En-tête
    c.setFont("Helvetica-Bold", 7)
    c.setFillColorRGB(*bgColor)
    c.rect(page_ctx.margin_left, y - 8, sum(col_widths), 8, fill=1, stroke=1)
    c.setFillColorRGB(0, 0, 0)
    

    header_x_offset = page_ctx.margin_left
    for header in article_table:
        c.drawString(header_x_offset + 2, y - 6, header.header)
        header_x_offset += header.width

    y -= 8
    c.setFont("Helvetica", 7)
    
    # Articles
    article_avec_commentaires = ""
    default_line_height = 12
    nb_lignes, nb_articles = 0, 0
    for article in data_ctx.articles:
        if article.get('code_article', '').startswith('$$'):
            continue
        # draw_anchor(c, page_ctx.margin_left, y)

        if article.get("code_article", "").startswith("$"):
            article_avec_commentaires += article.get("description","")+"<br/>"
            continue
        nbline_in_desc = len(article_avec_commentaires.split("<br/>"))
        line_height = nbline_in_desc * default_line_height
        translate_y = 0

        if with_lines:
            c.rect(page_ctx.margin_left, y - line_height, sum(col_widths), line_height)
        for x in col_x[1:]:
            c.line(x, y, x, y - line_height)
        if nbline_in_desc > 1:
            # on descend d'autant de lignes que de commentaires.
            # Attention ! La dernière ligne est la ligne de l'article
            translate_y = -(nbline_in_desc-1)*default_line_height
            y += translate_y

        c.drawString(article_table["code_article"].x + 1, y - 8, str(article.get('code_article', '')))

        c.setFont("Helvetica", 6.5)
        if article_avec_commentaires:  # Ajoute les commentaires dans la même case que l'article mais
            # juste avant
            y_description = y - translate_y
            for description in article_avec_commentaires.split("<br/>")[:-1]:
                c.drawString(col_x[1] + 2, y_description - 8, description)
                y_description -= default_line_height
            article_avec_commentaires = ""

        c.drawString(article_table["description"].x + 2, y - 8, article.get('description', ''))
        # TODO: créer une fonction draw_composition pour écrire la composition de l'article courant
        c.setFont("Helvetica", 7)
        
        tx_remise = article.get('tx_remise', 0)
        if tx_remise > 0:
            c.drawRightString(article_table["tx_remise"].x - 4, y - 8, str(tx_remise))

        quantity = article.get('quantite', 0)
        if quantity > 0:
            c.drawRightString(article_table["quantite"].x - 4, y - 8, str(quantity))
        
        pubase_ht = article.get('pubase_ht', 0)
        if pubase_ht > 0:
            c.drawRightString(article_table["pubase_ht"].x - 2, y - 8, f"{pubase_ht:.2f}")

        pu = article.get('valartht', 0) # prix unitaire HT après remise
        if pu > 0:
            c.drawRightString(article_table["pu_ht"].x - 2, y - 8, f"{pu:.2f}")

        montant_ht = article.get('totart', 0) # prix unitaire HT après remise
        if montant_ht > 0:
            c.drawRightString(article_table["montant_ht"].x - 2, y - 8, f"{montant_ht:.2f}")

        # montant = article.get('montant', 0)
        # if montant > 0:
        #     c.drawRightString(col_x[5] + col_widths[5] - 2, y - 8, f"{montant:.2f}")
        # totartbrut = article.get('totartbrut', None)
        # if totartbrut is not None:
        #     c.drawRightString(col_x[5] + col_widths[5] - 2, y - 8, f"{totartbrut:.2f}")
        
        tva_code = -1
        taxes = article.get("taxes", [])
        for taxe in taxes:
            if taxe.get("nom", "") == "TVA":
                tva_taux = taxe.get("taux", 0)
                tva_code = taxe.get("code", -1)

        # if tva_code > -1:
        #     c.drawCentredString((article_table["tva_code"].x + article_table["tva_code"].width) / 2, y - 8, str(tva_code))
        if tva_code > -1:
            c.drawCentredString(article_table["tva_code"].x+article_table["tva_code"].width/2, y - 8, str(tva_code))
        
        nb_lignes += 1
        nb_articles += quantity
        y -= default_line_height
    
    y -= 8
    c.setLineWidth(1)
    
    # Note NF525
    c.setFont("Helvetica", 6)
    version = data_ctx.pied.get('version', '')
    c.drawString(page_ctx.margin_left, y, f"{nb_lignes} ligne{'s' if nb_lignes>1 else ''} / {nb_articles} article{'s' if nb_articles>1 else ''}")
    # c.drawString(page_ctx.margin_left, y, f"(NF525) Version {version} / Nbr lignes: {nb_lignes}")
    
    # y -= 10
    # c.setFont("Helvetica-Bold", 8)
    # c.drawRightString(page_ctx.width - page_ctx.margin_right, y, 
    #                  "LES MONTANTS SONT EXPRIMES EN EUROS")
    
    y_bottom = y - 10
    height = y_top - y_bottom
    
    bbox = BoundingBox(page_ctx.margin_left, y_bottom, sum(col_widths), height, "Tableau_Articles")
    page_ctx.layout_manager.add(bbox)
    
    return y - 60, bbox


def draw_totaux(page_ctx: PageContext, data_ctx: DataContext, y_start: float) -> Tuple[float, BoundingBox]:
    """Dessine le tableau des totaux et retourne (y_final, empattement)"""
    c = page_ctx.canvas
    y = y_start
    y_top = y
    
    totaux_x = page_ctx.width - page_ctx.margin_right - 140
    totaux_width = 140
    
    c.setLineWidth(0.5)
    c.setFont("Helvetica", 8)
    
    pied = data_ctx.pied
    total_ht = pied.get('total_HT', 0)
    total_tva = pied.get('total_TVA', 0)
    total_ttc = pied.get('total_TTC', 0)
    
    # TOTAL H.T
    c.drawString(totaux_x + 5, y, "TOTAL H.T")
    c.drawRightString(totaux_x + totaux_width - 2, y, f"{total_ht:.2f} €")
    # c.rect(totaux_x, y - 2, totaux_width, 10)
    y -= 12
    
    # TOTAL T.V.A
    c.drawString(totaux_x + 5, y, "TOTAL TVA")
    c.drawRightString(totaux_x + totaux_width - 2, y, f"{total_tva:.2f} €")
    # c.rect(totaux_x, y - 2, totaux_width, 10)
    y -= 12
    
    # Ligne vide
    y -= 12
    
    # Ligne vide
    y -= 12
    
    # TOTAL T.T.C
    c.setFont("Helvetica-Bold", 9)
    c.drawString(totaux_x + 5, y, "TOTAL T.T.C")
    c.drawRightString(totaux_x + totaux_width - 2, y, f"{total_ttc:.2f} €")
    # c.rect(totaux_x, y - 2, totaux_width, 10)
    y -= 12
    
    # RESTE A PAYER
    c.setFont("Helvetica-Oblique", 7)
    c.drawString(totaux_x + 5, y, "RESTE A PAYER")
    c.drawRightString(totaux_x + totaux_width - 2, y, "0,00 €")
    # c.rect(totaux_x, y - 2, totaux_width, 10)
    
    c.setLineWidth(1)
    
    y_bottom = y - 2
    height = y_top - y_bottom
    
    bbox = BoundingBox(totaux_x - 5, y_bottom, totaux_width + 5, height, "Colonne_Totaux")
    page_ctx.layout_manager.add(bbox)
    
    return y, bbox


def draw_tva_table(page_ctx: PageContext, data_ctx: DataContext, y_position: float) -> BoundingBox:
    """Dessine le tableau TVA et retourne son empattement"""
    c = page_ctx.canvas
    
    tva_x = page_ctx.margin_left
    tva_y = y_position + 35
    y_top = tva_y + 10
    tva_width = 140
    
    c.setLineWidth(0.5)
    c.setFont("Helvetica-Bold", 7)
    
    # En-tête
    c.setFillColorRGB(*bgColor)
    c.rect(tva_x + 7, tva_y - 2, tva_width, 10, fill=1, stroke=1)
    c.setFillColorRGB(0, 0, 0)
    
    c.drawString(tva_x, tva_y, "T")
    c.drawString(tva_x + 15, tva_y, "Base")
    c.drawString(tva_x + 50, tva_y, "Tx (%)")
    c.drawString(tva_x + 85, tva_y, "Mt TVA")
    c.drawString(tva_x + 120, tva_y, "Base+TVA")
    
    tva_y -= 12
    c.setFont("Helvetica", 7)
    
    # Lignes TVA
    taxes = data_ctx.pied.get('taxes', [])
    for taxe in taxes[:4]:
        c.drawString(tva_x, tva_y, str(taxe.get('indice', '')))
        c.drawString(tva_x + 17, tva_y, f"{taxe.get('base_HT', 0):.2f}")
        c.drawString(tva_x + 52, tva_y, f"{taxe.get('taux', 0):.2f}")
        c.drawString(tva_x + 87, tva_y, f"{taxe.get('montant', 0):.2f}")
        base_ttc = taxe.get('base_HT', 0) + taxe.get('montant', 0)
        c.drawString(tva_x + 122, tva_y, f"{base_ttc:.2f}")
        c.rect(tva_x + 7, tva_y - 2, tva_width, 10)
        tva_y -= 12
    
    # Lignes vides restantes
    for i in range(len(taxes), 4):
        c.rect(tva_x + 7, tva_y - 2, tva_width, 10)
        tva_y -= 12
    
    c.setLineWidth(1)
    
    y_bottom = tva_y
    height = y_top - y_bottom
    
    bbox = BoundingBox(tva_x, y_bottom, tva_width + 7, height, "Tableau_TVA")
    page_ctx.layout_manager.add(bbox)
    return bbox


def draw_reglements(page_ctx: PageContext, data_ctx: DataContext) -> Tuple[BoundingBox, BoundingBox]:
    """Dessine les tableaux de règlements et retourne (bbox_gauche, bbox_droite)"""
    c = page_ctx.canvas
    y = 160  # Position fixe
    
    c.setLineWidth(0.5)
    c.setFont("Helvetica-Bold", 8)
    
    # Tableau Règlement/Échéance (gauche)
    bonus_box_width = 250
    bonus_x = page_ctx.margin_left
    bonus_y = y - 22
    
    c.rect(bonus_x, bonus_y, bonus_box_width, 22, fill=0, stroke=1)
    
    c.setFont("Helvetica", 7)
    c.setFillColorRGB(*bgColor)
    c.rect(bonus_x + 2, y - 9, 28, 8, fill=1, stroke=1)
    c.setFillColorRGB(0, 0, 0)
    c.drawString(bonus_x + 4, y - 7, "Réglement :")
    
    reglements = data_ctx.reglements
    if reglements:
        mode_reglement = reglements[0].get('mode', 'COMPTANT')
        c.drawString(bonus_x + 32, y - 7, mode_reglement)
    
    c.setFillColorRGB(*bgColor)
    c.rect(bonus_x + 2, y - 19, 28, 8, fill=1, stroke=1)
    c.setFillColorRGB(0, 0, 0)
    c.drawString(bonus_x + 4, y - 17, "Echéance :")
    
    date_formattee = formater_date(data_ctx.fac_date)
    c.drawString(bonus_x + 32, y - 17, date_formattee)
    
    bbox_gauche = BoundingBox(bonus_x, bonus_y, bonus_box_width, 22, "Tableau_Reglement")
    page_ctx.layout_manager.add(bbox_gauche)
    
    # Tableau Paiements (droite)
    pay_width = 250
    pay_x = page_ctx.width - page_ctx.margin_right - pay_width
    pay_y_top = 160
    pay_y = pay_y_top
    
    col_mp = 20
    col_desc = 140
    col_montant = 45
    col_echeance = 45
    
    c.setFont("Helvetica-Bold", 7)
    c.setFillColorRGB(*bgColor)
    c.rect(pay_x, pay_y - 2, pay_width, 10, fill=1, stroke=1)
    c.setFillColorRGB(0, 0, 0)
    
    c.drawString(pay_x + 2, pay_y, "MP")
    c.drawString(pay_x + 25, pay_y, "Comm./N°Chèque/Banque")
    c.drawString(pay_x + 165, pay_y, "Montant")
    c.drawString(pay_x + 215, pay_y, "Echéance")
    
    pay_y -= 12
    c.setFont("Helvetica", 7)
    
    # Lignes de paiement
    for reglement in reglements[:4]:
        c.rect(pay_x, pay_y - 2, col_mp, 10, fill=0, stroke=1)
        c.rect(pay_x + col_mp, pay_y - 2, col_desc, 10, fill=0, stroke=1)
        c.rect(pay_x + col_mp + col_desc, pay_y - 2, col_montant, 10, fill=0, stroke=1)
        c.rect(pay_x + col_mp + col_desc + col_montant, pay_y - 2, col_echeance, 10, fill=0, stroke=1)
        
        c.drawString(pay_x + 2, pay_y, str(reglement.get('code', '')))
        c.drawString(pay_x + col_mp + 2, pay_y, reglement.get('mode', ''))
        c.drawRightString(pay_x + col_mp + col_desc + col_montant - 2, pay_y, 
                         f"{reglement.get('montant', 0):.2f}")
        c.drawString(pay_x + col_mp + col_desc + col_montant + 2, pay_y, date_formattee)
        
        pay_y -= 10
    
    # Lignes vides restantes
    for i in range(len(reglements), 4):
        c.rect(pay_x, pay_y - 2, col_mp, 10, fill=0, stroke=1)
        c.rect(pay_x + col_mp, pay_y - 2, col_desc, 10, fill=0, stroke=1)
        c.rect(pay_x + col_mp + col_desc, pay_y - 2, col_montant, 10, fill=0, stroke=1)
        c.rect(pay_x + col_mp + col_desc + col_montant, pay_y - 2, col_echeance, 10, fill=0, stroke=1)
        c.drawString(pay_x + col_mp + col_desc + col_montant + 2, pay_y, "__/__/____")
        pay_y -= 10
    
    c.setLineWidth(1)
    
    pay_y_bottom = pay_y
    pay_height = pay_y_top - pay_y_bottom + 8
    
    bbox_droite = BoundingBox(pay_x, pay_y_bottom, pay_width, pay_height, "Tableau_Paiements")
    page_ctx.layout_manager.add(bbox_droite)
    
    return bbox_gauche, bbox_droite


def draw_footer(page_ctx: PageContext, data_ctx: DataContext) -> BoundingBox:
    """Dessine le bas de page et retourne son empattement"""
    c = page_ctx.canvas
    y = 65
    y_top = 65
    
    c.setFont("Helvetica", 6)
    c.drawString(page_ctx.margin_left, y, 
                "Le consommateur bénéficie de la garantie légale de conformité de 2 ans pour les achats de biens.")
    y -= 8
    c.drawString(page_ctx.margin_left, y, 
                "Nos conditions générales de vente sont librement consultables sur notre site internet et en magasin")
    y -= 8
    
    # Infos RCS et TVA
    emetteur = data_ctx.emetteur
    rcs = emetteur.get('rcs', '')
    tva_intra = emetteur.get('tvaintra', '')
    codnaf = emetteur.get('codnaf', '')
    
    if rcs:
        info_text = f"{rcs}"
        if codnaf:
            info_text += f" / APE: {codnaf}"
        if tva_intra:
            info_text += f" / N° TVA Intra. : {tva_intra}"
        c.drawString(page_ctx.margin_left, y, info_text)
    
    y -= 8
    
    # # Page
    # c.setFont("Helvetica-Bold", 8)
    # c.drawRightString(page_ctx.width - page_ctx.margin_right, 55, "Page : 1 / 1")
    
    # c.setLineWidth(0.5)
    # c.rect(page_ctx.width - page_ctx.margin_right - 80, 40, 80, 20)
    # c.setLineWidth(1)
    
    # c.drawCentredString(page_ctx.width - page_ctx.margin_right - 40, 50, "Facture")
    # c.drawCentredString(page_ctx.width - page_ctx.margin_right - 40, 43, 
    #                    "Paiement TVA d'après les débits")
    
    y_bottom = 40
    height = y_top - y_bottom
    
    bbox = BoundingBox(page_ctx.margin_left, y_bottom, page_ctx.textwidth, height, "Footer")
    page_ctx.layout_manager.add(bbox)
    return bbox


def draw_codes_barres(page_ctx: PageContext, data_ctx: DataContext, facture_bbox: BoundingBox) -> Tuple[BoundingBox, BoundingBox]:
    """Dessine le QR code et le code-barres et retourne (bbox_qr, bbox_barcode)"""
    c = page_ctx.canvas
    
    qr_x = page_ctx.width - page_ctx.margin_right - 180
    qr_y = facture_bbox.y + facture_bbox.height + 5
    qr_size = 35
    barcode_width = 80
    barcode_height = 35
    
    try:
        # QR Code
        try:
            qr = qrcode.QRCode(version=1, box_size=3, border=1)
            qr.add_data(data_ctx.fac_num)
            qr.make(fit=True)
            qr_img = qr.make_image(fill_color="black", back_color="white")
            
            temp_qr = tempfile.NamedTemporaryFile(delete=False, suffix='.png')
            qr_img.save(temp_qr.name)
            temp_qr.close()
            
            c.drawImage(temp_qr.name, qr_x, qr_y, width=qr_size, height=qr_size, 
                       preserveAspectRatio=True, mask='auto')
            
            try:
                os.remove(temp_qr.name)
            except:
                pass
        except Exception as e:
            print(f"Erreur génération QR code: {e}")
        
        bbox_qr = BoundingBox(qr_x, qr_y, qr_size, qr_size, "QR_Code")
        page_ctx.layout_manager.add(bbox_qr)
        
        # Code-barres
        barcode_path = generate_barcode(data_ctx.fac_num + "0" * 10)
        bbox_barcode = None
        
        if barcode_path:
            barcode_x = qr_x + 45
            c.drawImage(barcode_path, barcode_x, qr_y, width=barcode_width, height=barcode_height, 
                       preserveAspectRatio=True, mask='auto')
            bbox_barcode = BoundingBox(barcode_x, qr_y, barcode_width, barcode_height, "Barcode")
            page_ctx.layout_manager.add(bbox_barcode)
            try:
                os.remove(barcode_path)
            except:
                pass
        
        return bbox_qr, bbox_barcode
        
    except Exception as e:
        print(f"Erreur génération codes: {e}")
        return None, None


# ========== FONCTION PRINCIPALE ==========

def generer_facture_depuis_json(data_json, fichier_sortie, logo_path=None, debug_logo=False, check_overlaps=True):
    """
    Génère une facture PDF à partir de données JSON
    
    Args:
        data_json: Dictionnaire Python ou chemin vers fichier JSON
        fichier_sortie: Chemin du fichier PDF de sortie
        logo_path: Chemin optionnel vers le fichier logo (PNG/JPG)
        debug_logo: Si True, affiche un cadre autour du logo
        check_overlaps: Si True, vérifie les chevauchements d'éléments
    """
    # Charger les données si c'est un fichier
    if isinstance(data_json, str):
        with open(data_json, 'r', encoding='utf-8') as f:
            data = json.load(f)
    else:
        data = data_json

    data_ctx = DataContext(data=data)

    # Création des contextes
    c = canvas.Canvas(fichier_sortie, pagesize=A4)
    width, height = A4
    articles_y = 507 # 519.19
    totaux_y = 60 * mm

    if data_ctx.emetteur_logo_path:
        logo_path = data_ctx.emetteur_logo_path

    page_ctx = PageContext(
        canvas=c,
        width=width,
        height=height,
        margin_left=10 * mm,
        margin_right=10 * mm,
        margin_top=20 * mm,
        margin_bottom=20 * mm,
        logo_path=logo_path,
        debug_logo=debug_logo
    )
    
    
    # Génération de la facture par sections
    logo_bbox = draw_logo(page_ctx)

    facture_bbox = draw_numero_piece(page_ctx, data_ctx)

    entete_client_bbox = draw_entete_client(page_ctx, data_ctx)

    if data_ctx.emetteur_legacy:
        y, emetteur_bbox = draw_emetteur_legacy_header(page_ctx, data_ctx, logo_bbox.y - 25)
    else:
        y, emetteur_bbox = draw_emetteur_info(page_ctx, data_ctx, logo_bbox.y - 25)
    
    y, info_fact_bbox = draw_info_facturation_v2(page_ctx, data_ctx, y)
    
    y, articles_bbox = draw_articles_table(page_ctx, data_ctx, articles_y, True)
    
    y, totaux_bbox = draw_totaux(page_ctx, data_ctx, totaux_y)
    
    # tva_bbox = draw_tva_table(page_ctx, data_ctx, y)
    
    # reglement_bbox, paiement_bbox = draw_reglements(page_ctx, data_ctx)
    
    footer_bbox = draw_footer(page_ctx, data_ctx)
    
    qr_bbox, barcode_bbox = draw_codes_barres(page_ctx, data_ctx, facture_bbox)
    
    # Vérification des chevauchements
    if check_overlaps:
        overlaps = page_ctx.layout_manager.report_overlaps()
        if overlaps:
            print("\n=== RAPPORT DES CHEVAUCHEMENTS ===")
            for msg in overlaps:
                print(msg)
            print("==================================\n")

    if page_ctx.debug_logo:
        c.setStrokeColorRGB(1, 0, 0)
        c.setLineWidth(0.5)
        c.rect(page_ctx.margin_left, page_ctx.margin_bottom,
               page_ctx.width-page_ctx.margin_right-page_ctx.margin_left,
               page_ctx.height-page_ctx.margin_top-page_ctx.margin_bottom)
        c.setStrokeColorRGB(0, 0, 0)
        c.setLineWidth(1)
    
    # Sauvegarder
    c.save()
    print(f"Facture générée : {fichier_sortie}")


# ========== CLI avec Typer ==========

app = typer.Typer(help="Générateur de factures PDF à partir de données JSON")

@app.command()
def main(
    json_file: Path = typer.Argument(
        ...,
        help="Fichier JSON d'entrée contenant les données de la facture",
        exists=True,
        file_okay=True,
        dir_okay=False,
        readable=True,
    ),
    output_dir: Path = typer.Option(
        ...,
        "--output-dir",
        help="Répertoire de sortie pour le fichier PDF (par défaut: répertoire courant)",
    ),
    logo: Path = typer.Option(
        None,
        "--logo",
        help="Chemin vers le fichier logo (PNG/JPG)",
        exists=True,
        file_okay=True,
        dir_okay=False,
        readable=True,
    ),
    debug_logo: bool = typer.Option(
        False,
        "--debug-logo",
        help="Affiche un cadre rouge autour du logo pour visualiser l'espace occupé",
    ),
    check_overlaps: bool = typer.Option(
        True,
        "--check-overlaps/--no-check-overlaps",
        help="Vérifie les chevauchements d'éléments sur la page",
    ),
):
    """
    Génère une facture PDF à partir d'un fichier JSON.

    Exemple:
        python generer_facture_json.py facture.json -o facture.pdf
        python generer_facture_json.py facture.json --logo logo.png
        python generer_facture_json.py facture.json --logo logo.png --debug-logo
    """
    # Charger les données JSON
    try:
        with open(json_file, 'r', encoding='cp850') as f:
            data = json.load(f)
    except json.JSONDecodeError as e:
        typer.echo(f"Erreur: Fichier JSON invalide - {e}", err=True)
        raise typer.Exit(code=1)
    except Exception as e:
        typer.echo(f"Erreur lors de la lecture du fichier: {e}", err=True)
        raise typer.Exit(code=1)

    # Déterminer le nom du fichier de sortie
    fac_num = data.get('fac_num', 'UNKNOWN')
    if output_dir is not None:
        output_dir = Path(output_dir)
        if output_dir.is_dir():
            output = output_dir / f"JSON-{fac_num}.pdf"
    else:
        output = f"JSON-{fac_num}.pdf"

    # Générer la facture
    try:
        generer_facture_depuis_json(
            data,
            str(output),
            logo_path=str(logo) if logo else None,
            debug_logo=debug_logo,
            check_overlaps=check_overlaps
        )
        typer.echo(f"[OK] Facture générée avec succès: {output}")
    except Exception as e:
        typer.echo(f"[ERREUR] Lors de la génération de la facture: {e}", err=True)
        raise typer.Exit(code=1)


if __name__ == "__main__":
    app()
