from datetime import datetime
import json
from pathlib import Path
import tempfile

import numpy as np
import pandas as pd
# Voir append et reindex ici https://pbpython.com/excel-pandas-comp.html

from . import etats


def get_prct(a, b):
    return (b-a)/a * 100
            

def inserer_sous_totaux(df):
    # Calcul des sous totaux par familles
    top_fam = df.index.get_level_values(0).unique().to_list()
    sub_df = []
    for k in top_fam:
        rslt = df.loc[k].sort_index().copy()
        ## Faire le cumul des ventes
        rslt.loc[f"total {k}"] = rslt.sum()
        sub_df.append(rslt.set_index(rslt.index.map(lambda x: (k,x)))) # reindexer en 

    return pd.concat(sub_df)


def round_if_not_inf(x):
    return round(x) if x != np.inf else "-"


def cumuler_ventes_par_familles(multi_df, dates, soustotal=False, depot_description=""):
    weeks    = multi_df.index.get_level_values(0).unique().to_list()
    familles = multi_df.index.get_level_values(1).unique().to_list()
    weeks_colname = [f"S{w}" for w in weeks]

    rslt = pd.DataFrame(index=familles)
    for iweek, week in enumerate(weeks):
        dt = dates[iweek]

        rslt[dt]          = multi_df[multi_df.date==dt].groupby("famille")["qt"].sum()
        rslt[f"S{week}"]  = multi_df.loc[week].groupby("famille")["qt"].sum()
        # QP = vente(fam,sem)/vente(all,sem)
        rslt[f"QP{week}"] = rslt[f"S{week}"]/multi_df.loc[week]["qt"].sum()*100

    rslt["Evol"]  = 0 #np.nan #get_prct(rslt[dates[1]], rslt[dates[0]])
    rslt["SEvol"] = 0 #np.nan #get_prct(rslt[f"S{weeks[1]}"], rslt[f"S{weeks[0]}"])
    
    total_ventes = rslt.sum().rename("total")
    total_ventes = total_ventes.rename(("total",f"total {depot_description}")) # Mettre un multi-index

    rslt = rslt.set_index(rslt.index.map(lambda x: (x[0],x)))
    if soustotal:
        rslt = inserer_sous_totaux(rslt)
            
    ## Faire le cumul des ventes
    rslt = rslt.append(total_ventes)
    #subtotals = rslt.groupby(level=0).sum()

    #rslt.loc["total"] = rslt.sum()

    rslt.fillna(0, inplace=True)
    # Calcule la progression en pourcentage
    rslt.loc[:, "Evol"]  = get_prct(rslt[dates[0]], rslt[dates[1]])
    rslt.loc[:, "SEvol"] = get_prct(rslt[weeks_colname[0]], rslt[weeks_colname[1]])

    # Transforme le nom des colonnes de type datetime en str
    rslt.rename({dt :datetime.strftime(dt, "%d-%m-%Y") for dt in dates}, axis=1, inplace=True)
    dates_str = [datetime.strftime(dt, "%d-%m-%Y") for dt in dates]
    # Reordonner les colonnes
    ordres = dates_str + ["Evol"] + ",".join([f"S{w},QP{w}" for w in weeks]).split(",") + ["SEvol"]

    for col in ["Evol", "QP1", "QP2", "SEvol"]:
        rslt.loc[:,col] = rslt[col].fillna(0).map(lambda x: round_if_not_inf(x) )#.fillna(0)#.astype(int)
    # rslt.replace([np.inf, -np.inf], np.nan, inplace=True)

    # retourner une dataframe reordonnée
    rslt.index.set_names(["top_fam", "famille"], inplace=True)

    # Convertir en entier
    # Attention à mettre I majuscule pour gérer les nullable
    colonnes = rslt.columns.to_list()#.remove("designation")
    # for cval in colonnes:
    for cval in dates_str + [f"S{w}" for w in weeks]:
        rslt.loc[pd.IndexSlice[:,:], cval] = rslt.loc[pd.IndexSlice[:,:], cval].astype("Int64")
    return rslt[ordres].sort_index()



def indicateur_vente(multi_df, cumul, dates, depot_description):
    """Compte le nombre de clients et le nombre de vente moyen par client"""

    weeks    = multi_df.index.get_level_values(0).unique().to_list()
    
    rslt = pd.DataFrame()
    for iweek, week in enumerate(weeks):
        dt = dates[iweek]
        rslt.loc["Nombre clients", dt] = multi_df.loc[multi_df.date==dt,"numfac"].unique().shape[0]
        rslt.loc["Nombre clients", f"S{week}"] = multi_df.loc[week,"numfac"].unique().shape[0]

    dates_str = [datetime.strftime(dt, "%d-%m-%Y") for dt in dates]
    rslt.rename({dt :datetime.strftime(dt, "%d-%m-%Y") for dt in dates}, axis=1, inplace=True)
    
    # ======================
    # Calcul de l’indicateur de vente (nombre moyen de produits acheté par client)
    colonnes = cumul.columns[[0,1,3,5]] # recuperer colonnes avec dates et semaines
    IV = cumul.loc[("total",f"total {depot_description}"), colonnes] / rslt[colonnes]
    IV = IV.rename({"Nombre clients":"IV"})#.applymap(lambda x: round(x,2) )
    rslt = rslt.append(IV)

    #rslt[["Evol","SEvol"]] = np.nan
    rslt.loc[:,"Evol"]  = get_prct(rslt.iloc[:,0], rslt.iloc[:,1])
    rslt.loc[:,"SEvol"] = get_prct(rslt.iloc[:,3], rslt.iloc[:,4])

    #for col in ["Evol", "QP1", "QP2", "SEvol"]:
    #    rslt.loc[:,col] = rslt[col].fillna(0).map(lambda x: round_if_not_inf(x) )#.fillna(0)#.astype(int)

    rslt.loc[:,"col1"] = rslt.index.to_list()
    return rslt[["col1"]+dates_str + ["Evol"]+[f"S{w}" for w in weeks]+["SEvol"]]

def lire_ventes_par_depot(args, data)->pd.DataFrame:
    ventes_depots = data.copy()
    ventes_depots["famille"] = ventes_depots["code"].apply(lambda x: x[:args.fam_len])
    ventes_depots["date"] = pd.to_datetime(ventes_depots["date"], dayfirst=True, format="%d/%m/%Y")
    ventes_depots["week"] = ventes_depots["date"].apply(lambda d: d.isocalendar()[1])

    ventes_depots = ventes_depots.set_index(['numdep','week', 'famille']).sort_index()
    # consolide = ventes_depots.set_index(['week', 'famille']).sort_index()
    return ventes_depots

def get_familles_non_vendues(designation_csv:str, data:pd.DataFrame):
    """Compare la liste des familles vendues à la liste"""
    design = pd.read_csv(designation_csv, sep=";", encoding="cp850")
    # Ne garder que les codes familles complets car dans le fichier csv il n’y a pas encore de
    # notion de niveau de détail
    familles = [fam for fam in  design.famille if len(fam)==4] 

    # # numdep        date      numfac  code   fam  qt
    # # 1 il faut filtrer par depots
    df = pd.DataFrame()
    for dep in data["numdep"].unique():
        famille_non_vendues = set(familles).difference( set(data.loc[data.numdep==dep,"fam"].unique()))
        df = pd.concat(
                [df,
                pd.DataFrame(
                    data={
                        "code": list(famille_non_vendues),
                        "qt": 0,
                        "numdep": dep,
                        "top_fam": [fam[0] for fam in famille_non_vendues]
                    }
                )],
            ignore_index=True,
        )

    df = df.reindex(columns=set(df.columns.to_list()).union(data.columns.to_list()))

    return df


def ventes_consolides(ventes_depots):
    # supprime l’index "numdep"
    return ventes_depots.reset_index().set_index(['week', 'famille']).sort_index()

def get_depot_IDs(ventes_depots)->list:
    return ventes_depots.index.get_level_values("numdep").unique().to_list()

def ajoute_colonne_designation(designation_csv, data):
    design = pd.read_csv(designation_csv, sep=";", encoding="cp850")
    pdIdx = pd.IndexSlice # permet de faire une selection sur le mulitiindex
    # for fam in data.index.get_level_values("top_fam").unique():
    for fam in data.index.get_level_values("famille").unique():
        if not fam.startswith("total"):
            data.loc[pdIdx[:,fam], "designation"] = design[design.famille==fam]["designation"].values[0]

    # Placer la colonne "designation" en premier
    return data[["designation"]+data.columns.to_list()[:-1]] 



def multicol(text, nbcols, align="l"):
    # f"\\SetCell[c={nbcols}]{{c}}{{{titre}}}"
    return f"\\multicolumn{{{nbcols}}}{{{align}}}{{{text}}}"

def lire_fichier(fichier):
    with open(fichier) as fp:
        return "".join(fp.readlines())

def generer_json(args, csvpath, colonnes_noms, colonnes, titre, newpage=False):
    nbcols = len(colonnes)
    sous_titres = [multicol("",2), multicol(args.dtref_dayname,3,"c"),multicol("Semaines",5,"c")]

    # TODO Ajouter la couleur
    couleurs = etats.styles.Couleurs.from_string(args.style)
    noms = [f"\\cmidrule[lr,{couleurs.bg},1pt]{{3-5}} \\cmidrule[lr=-0.4,{couleurs.bg},1pt]{{6-10}} \n"]
    noms = [noms[0]+colonnes_noms[0]] + colonnes_noms[1:]
    return {
            "style": args.style,
            "type": "LongListeSousTotal",
            "csv": str(csvpath),
            "colspec": "cX[l,3]cccrrrrr",
            "titre": True,
            "newpage": newpage,
            "header":[ [multicol(titre,nbcols,"c")], # Titre entete
                        sous_titres,                 # ligne 1 entete
                        noms                         # ligne 2 entete
                       ],
            "texte_avant": "\\textbf{QP}: proportion de vente de la famille par rapport au total",
            # "texte_apres": lire_fichier("texte.lipsum")
           }

def generer_json_famille_non_vendues(args, csvpath, colonnes_noms, colonnes, titre, newpage=False):
    # La premiere colonnes doit etre en gras
    # colonnes_noms[0] = "\\SetColumn{font=\\bfseries\\large\\sffamily}"+colonnes_noms[0]
    nbcols = len(colonnes)
    # Inverse les 2 dernières colonnes
    colonnes = colonnes[:-2] + colonnes[-2:][::-1]
    colonnes_noms = colonnes_noms[:-2] + colonnes_noms[-2:][::-1]
    return {
            "style": args.style,
            "type": "LongListeSousTotal",
            "csv": str(csvpath),
            "colspec": "cX[l,3]X[r,3]c",
            "titre": True,
            "header": [[multicol("Familles sans ventes sur les périodes d’étude",nbcols,"c")],
                        colonnes_noms],
            "colonnes": colonnes,
            "newpage": newpage,
            "legend": "Familles sans vente sur les périodes"
           }

def generer_json_IV(args, csvpath, colonnes_noms, colonnes, titre, newpage=False):
    # La premiere colonnes doit etre en gras
    colonnes_noms[0] = "\\SetColumn{font=\\bfseries\\large\\sffamily}"+colonnes_noms[0]
    return {
            "style":args.style,
            "type": "Liste",
            "csv": str(csvpath),
            "colspec": "crrrrrr",
            "header": [colonnes_noms],
            "colonnes": colonnes,
            "newpage": newpage
           }

def generer_sorties_tableau_familles_non_vendues(args, csvpath_prefix, titre, fam_not_sold):
    fam_designa = pd.read_csv(args.csvfam, encoding="cp850", sep=";")
    fam_designa = fam_designa[ fam_designa.famille.isin(fam_not_sold.code.to_list()) ]
    if fam_designa.shape[0]>0:
        # Reshape fam_designa pour la sortie csv
        nbrows = fam_designa.shape[0]//2
        df = pd.DataFrame(
                    {col:fam_designa.iloc[:nbrows,icol].to_list() for icol,col in enumerate(fam_designa.columns)}
                )
        if fam_designa.shape[0]%2==0:
            df = df.assign(
                        **{col+"_1":fam_designa.iloc[nbrows:,icol].to_list() for icol,col in enumerate(fam_designa.columns)}
                    )
        else:
            df = df.assign(
                        **{col+"_1":fam_designa.iloc[nbrows:-1,icol].to_list() for icol,col in enumerate(fam_designa.columns)}
                    )
            df = pd.concat([df,fam_designa.iloc[fam_designa.shape[0]-1].to_frame().T], ignore_index=True)

        csvpath = f"{csvpath_prefix}_nonvendu.csv"
        df.to_csv(csvpath, sep=";", encoding="utf8", na_rep=" ", index=False)

        return generer_json_famille_non_vendues(args, csvpath, 2*["Famille", "Désignation"],
                                                    df.columns.to_list(), titre, newpage=False)

    else:
        return None

def generer_les_sorties(args, csvpath_prefix, dates, itableau, data,
                        dict_to_json, titre, depot_description:str, fam_not_sold):
    """ Génère 2/3 fichiers CSV et retourne un dictionnaire contenant la configuration JSON des tableaux
    Parameters
    ----------
    args: ParserArguments
        Arguments passés au script python
    csvpath_prefix: str, Pathlib
        prefix des noms de fichiers des CSV générés
    dates:
    itableau: int
        identificateur des tableaux générés. Dans le JSON, un tableau est identifié par "tableau_x", où
        est incrémenté
    titre: str
    """
    cumul = cumuler_ventes_par_familles(data, dates, soustotal=args.with_soustotal,
                                        depot_description=depot_description)
    # # ajoute les familles non vendue à cumul
    # for _, row in fam_not_sold.iterrows():
    #     cumul.loc[ (row["top_fam"],row["code"]),:] = 0
    # cumul.sort_index(inplace=True)
    IV_cumul = indicateur_vente(data, cumul , dates,  depot_description)

    cumul = ajoute_colonne_designation(args.csvfam, cumul)

    # Reformater pour sortie csv
    csvpath = f"{csvpath_prefix}.csv"
    cumul.reset_index(level="famille", col_level=1, inplace=True)
    dates_str = [datetime.strftime(dt, "%d-%m-%Y") for dt in dates]
    # cumul.iloc[:,1] = cumul.iloc[:,1].astype(int)
    # cumul.iloc[:,2] = cumul.iloc[:,2].astype(int)
    # cumul = cumul.applymap(lambda x: "" if x==0 else x)
    cumul_header = []
    for name in cumul.columns.to_list():
        if name.startswith("QP"):
            cumul_header.append("QP")
        elif name.endswith("SEvol"):
            cumul_header.append("Évol.")
        else:
            cumul_header.append(name)

    if args.cache_zero:
        cumul.replace([0]," ").to_csv(csvpath, sep=";", encoding="utf8", index=False, na_rep=" ")
    else:
        cumul.to_csv(csvpath, sep=";", encoding="utf8", index=False, na_rep=" ")
    dict_to_json[f"tableau_{itableau}"] = generer_json(args, csvpath, cumul_header,
                                                       cumul.columns.to_list(),
                                                       titre,
                                                       True)

    itableau += 1
    csvpath = f"{csvpath_prefix}_IV.csv"
    IV_cumul.to_csv(csvpath, sep=";", encoding="utf8", index=False)
    colonnes_noms = [""] + IV_cumul.columns.to_list()[1:]
    dict_to_json[f"tableau_{itableau}"] = generer_json_IV(args, csvpath, colonnes_noms,
                                                          IV_cumul.columns.to_list(),
                                                          titre,
                                                          False)

    output = generer_sorties_tableau_familles_non_vendues(args, csvpath_prefix, titre, fam_not_sold)
    if output:
        itableau += 1
        dict_to_json[f"tableau_{itableau}"] = output
    return itableau + 1


def exec(args):
    # ref_date   =  datetime.strptime( arg"ref"], "%d-%m-%Y")
    # other_date = ref_date - timedelta(days=7)
    dates = [ args.dtother, args.dtref]

    data_brute = pd.read_csv(args.csvpath, sep=args.sep)

    ventes_depots = lire_ventes_par_depot(args, data_brute) 
    consolide = ventes_consolides(ventes_depots)
    familles_non_vendues = get_familles_non_vendues(args.csvfam, data_brute)
    del(data_brute)

    dict_to_json = {}
    itableau = 0

    depots_IDs =  ventes_depots.index.get_level_values("numdep").unique()
    if args.depots == "tous":
        depots_to_print = depots_IDs
    else:
        depots_to_print = set(args.depots).intersection(depots_IDs)

    # Depot par dépots
    for idx, depID in enumerate(list(depots_to_print)):
        data = ventes_depots.loc[depID]
        fam_not_sold = familles_non_vendues[familles_non_vendues.numdep==depID]
        csvpath_prefix = args.tempdir / f"cumul_ventes_par_famille_depots_{idx}"
        titre = f"Comparatif des ventes par famille dépôt n {depID}"
        itableau = generer_les_sorties(args, csvpath_prefix, dates,
                                       itableau, data, dict_to_json, titre,
                                       f"magasin {depID}", fam_not_sold)

    # print("Consolide")
    # TODO consolide devrait correspondre à la somme des dépots imprimés pas tous ?
    if args.consolide:
        csvpath_prefix = args.tempdir / "cumul_ventes_par_famille_consolide"
        titre = "Comparatif des ventes par famille tous dépôts"
        itableau = generer_les_sorties(args, csvpath_prefix, dates,
                                       itableau, consolide, dict_to_json, titre,
                                       "tous magasins",
                                       familles_non_vendues)

    jsonfile = args.tempdir / "cumuler_ventes_par_famille.json"
    with open(jsonfile, "w") as fp:
        json.dump(dict_to_json, fp, indent=2)

    return args.tempdir / "cumuler_ventes_par_famille.json"

class CumulArgs:

    def __init__(self, fam_len:int, dtref:str, dtother:str,
                 csv:str, csvfam:str, sep:str,
                 consolide:bool,
                 depots=None,
                 cache_zero=None,
                 **kwargs) -> None:
        self.fam_len = fam_len or 1
        self.dtref   = datetime.strptime(dtref  , "%d-%m-%Y") or None
        self.dtother = datetime.strptime(dtother, "%d-%m-%Y") or None
        self.csvpath = csv or None
        self.csvfam  = csvfam or None
        self.consolide = consolide or False
        self.with_soustotal = self.fam_len>1
        self.sep = sep or ";"
        self.dtref_dayname = datetime.strftime(self.dtref, "%A") if self.dtref else None
        self.dtother_dayname = datetime.strftime(self.dtother, "%A") if self.dtother else None
        if self.dtref_dayname != self.dtother_dayname:
            raise ValueError(f"Les jours des dates ne sont pas identique {self.dtother:%A %d-%m-%Y} vs {self.dtref:%A %d-%m-%Y}")

        self.cache_zero = cache_zero or None
        if self.cache_zero is None:
            self.cache_zero = False
        self.depots = depots or None
        if self.depots is None:
            self.depots = []
        elif isinstance(depots, int):
            self.depots = [depots]
        elif isinstance(depots, list):
            self.depots = depots
        else:
            self.depots = "tous"


        for k,val in kwargs.items():
            setattr(self, k, val)
        # self.periode = periode.strip().replace(' ','') or None
        # if "=" not in self.periode:
        #     raise ValueError(f"periode [{self.periode}] ne respecte pas la syntaxe attendue: days=-7")
        #     self.periode = "days=-7"
        # per = self.periode.split("=")
        # if len(per) != 2:
        #     raise ValueError(f"periode [{self.periode}] ne respecte pas la syntaxe attendue: days=-7")
        # key, val = per
        # if key in ["days", "years"]:

        # self.dtother = self.dtref + timedelta(days=)
        # Attention ! tempfile.mkdtemp, vous devez gérer la suppression du dossier
        self.tempdir = Path(tempfile.mkdtemp(prefix="cumuler_ventes_par_famille")) 
